<?php
class AuthController {
    public static function login(): void {
        $body = Validator::getBody();
        Validator::make($body)->required('email', 'password')->email('email')->validate();

        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM usuarios WHERE email = ? AND activo = 1 LIMIT 1");
        $stmt->execute([$body['email']]);
        $user = $stmt->fetch();

        if (!$user || !password_verify($body['password'], $user['password_hash'])) {
            if (class_exists('AuditService')) AuditService::loginFallido($body['email']);
            Response::error('Credenciales incorrectas', 401);
        }

        $db->prepare("UPDATE usuarios SET ultimo_login = NOW() WHERE id = ?")->execute([$user['id']]);
        if (class_exists('AuditService')) AuditService::loginExitoso($user['id']);

        $payload = ['sub' => $user['id'], 'email' => $user['email'], 'rol' => $user['rol'], 'nombre' => $user['nombre_completo']];
        $token = JWT::generar($payload);

        Response::success([
            'token' => $token,
            'usuario' => [
                'id' => $user['id'], 'nombre_completo' => $user['nombre_completo'],
                'email' => $user['email'], 'rol' => $user['rol'],
                'departamento' => $user['departamento']
            ]
        ], 'Login exitoso');
    }

    public static function register(): void {
        $body = Validator::getBody();
        Validator::make($body)->required('nombre_completo', 'email', 'password')
            ->email('email')->minLength('password', 6)->validate();

        $db = getDB();
        $exists = $db->prepare("SELECT id FROM usuarios WHERE email = ?");
        $exists->execute([$body['email']]);
        if ($exists->fetch()) Response::error('El email ya está registrado', 409);

        $hash = password_hash($body['password'], PASSWORD_BCRYPT);
        $stmt = $db->prepare("INSERT INTO usuarios (uuid, nombre_completo, email, password_hash, rol, departamento, activo) VALUES (UUID(),?,?,?,?,?,1)");
        $stmt->execute([
            $body['nombre_completo'], $body['email'], $hash,
            in_array($body['rol'] ?? 'usuario', ['admin','supervisor','tecnico','usuario']) ? ($body['rol'] ?? 'usuario') : 'usuario',
            $body['departamento'] ?? null
        ]);
        $id = (int)$db->lastInsertId();
        if (class_exists('AuditService')) AuditService::usuarioCreado($id);
        Response::created(['id' => $id], 'Usuario registrado');
    }

    public static function me(): void {
        $payload = AuthMiddleware::verificar();
        $db = getDB();
        $stmt = $db->prepare("SELECT id, nombre_completo, email, rol, departamento, telefono, avatar_url, created_at, ultimo_login FROM usuarios WHERE id = ?");
        $stmt->execute([$payload['sub']]);
        $user = $stmt->fetch();
        if (!$user) Response::notFound('Usuario no encontrado');
        Response::success($user);
    }

    public static function refresh(): void {
        $payload = AuthMiddleware::verificar();
        $newToken = JWT::generar(['sub' => $payload['sub'], 'email' => $payload['email'], 'rol' => $payload['rol'], 'nombre' => $payload['nombre']]);
        Response::success(['token' => $newToken]);
    }

    public static function cambiarPassword(): void {
        $payload = AuthMiddleware::verificar();
        $body = Validator::getBody();
        Validator::make($body)->required('password_actual', 'password_nueva')->minLength('password_nueva', 6)->validate();

        $db = getDB();
        $stmt = $db->prepare("SELECT password_hash FROM usuarios WHERE id = ?");
        $stmt->execute([$payload['sub']]);
        $user = $stmt->fetch();

        if (!password_verify($body['password_actual'], $user['password_hash'])) {
            Response::error('Contraseña actual incorrecta', 401);
        }

        $hash = password_hash($body['password_nueva'], PASSWORD_BCRYPT);
        $db->prepare("UPDATE usuarios SET password_hash = ? WHERE id = ?")->execute([$hash, $payload['sub']]);
        if (class_exists('AuditService')) AuditService::cambioPassword($payload['sub']);
        Response::success(null, 'Contraseña actualizada');
    }
}
