<?php
class ConfiguracionController {

    public static function index(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $db = getDB();
        $grupo = $_GET['grupo'] ?? null;
        if ($grupo) {
            $stmt = $db->prepare("SELECT * FROM configuracion WHERE grupo = ? ORDER BY id");
            $stmt->execute([$grupo]);
        } else {
            $stmt = $db->query("SELECT * FROM configuracion ORDER BY grupo, id");
        }
        Response::success($stmt->fetchAll());
    }

    public static function grupos(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $db = getDB();
        
        $grupos = $db->query("SELECT DISTINCT grupo FROM configuracion ORDER BY grupo")->fetchAll(PDO::FETCH_COLUMN);
        
        if (!in_array('general', $grupos)) {
            self::ensureGeneralConfig($db);
            array_unshift($grupos, 'general');
        } else {
            $grupos = array_diff($grupos, ['general']);
            array_unshift($grupos, 'general');
        }
        
        Response::success(array_values($grupos));
    }

    /**
     * PUT /configuracion - Actualizar múltiples configuraciones
     */
    public static function actualizar(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $body = Validator::getBody();
        $db = getDB();

        if (empty($body['configuraciones']) || !is_array($body['configuraciones'])) {
            Response::error('Se requiere un array de configuraciones');
        }

        $stmt = $db->prepare("UPDATE configuracion SET valor = ?, updated_at = NOW() WHERE clave = ?");
        $stmtInsert = $db->prepare("INSERT INTO configuracion (clave, valor, grupo, descripcion, tipo) VALUES (?,?,?,?,?) ON DUPLICATE KEY UPDATE valor = VALUES(valor), updated_at = NOW()");
        $updated = 0;
        
        foreach ($body['configuraciones'] as $item) {
            if (!empty($item['clave'])) {
                $stmt->execute([$item['valor'] ?? '', $item['clave']]);
                if ($stmt->rowCount() === 0 && isset($item['grupo'])) {
                    $stmtInsert->execute([$item['clave'], $item['valor'] ?? '', $item['grupo'], $item['descripcion'] ?? '', $item['tipo'] ?? 'text']);
                }
                $updated++;
            }
        }

        try {
            $db->prepare("INSERT INTO log_actividad (usuario_id, accion, entidad_tipo, entidad_id, datos_nuevos, ip_address) VALUES (?,?,?,?,?,?)")
                ->execute([$payload['sub'], 'actualizar_configuracion', 'configuracion', null, json_encode($body['configuraciones']), $_SERVER['REMOTE_ADDR'] ?? null]);
        } catch(\Exception $e) {}

        Response::success(['updated' => $updated], 'Configuración actualizada');
    }
    
    /**
     * PUT /configuracion/{clave} - Actualizar una clave individual
     */
    public static function actualizarClave(string $clave): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $body = Validator::getBody();
        $db = getDB();
        
        $valor = $body['valor'] ?? '';
        
        $stmt = $db->prepare("UPDATE configuracion SET valor = ?, updated_at = NOW() WHERE clave = ?");
        $stmt->execute([$valor, $clave]);
        
        if ($stmt->rowCount() === 0) {
            $grupo = $body['grupo'] ?? 'general';
            $db->prepare("INSERT INTO configuracion (clave, valor, grupo, descripcion, tipo) VALUES (?,?,?,?,?)")
                ->execute([$clave, $valor, $grupo, $body['descripcion'] ?? '', $body['tipo'] ?? 'text']);
        }
        
        Response::success(null, "Configuración '$clave' actualizada");
    }
    
    /**
     * POST /configuracion/upload-logo - Subir logo de la organización
     */
    public static function uploadLogo(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $db = getDB();
        
        if (empty($_FILES['logo'])) {
            Response::error('No se recibió archivo');
        }
        
        $file = $_FILES['logo'];
        $allowed = ['image/png', 'image/jpeg', 'image/gif', 'image/svg+xml', 'image/webp'];
        $mime = mime_content_type($file['tmp_name']) ?: $file['type'];
        
        if (!in_array($mime, $allowed)) {
            Response::error('Formato no permitido. Use PNG, JPG, GIF, SVG o WebP');
        }
        
        if ($file['size'] > 2 * 1024 * 1024) {
            Response::error('El archivo no debe superar 2MB');
        }
        
        $uploadDir = UPLOAD_DIR . '/branding/';
        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $ext = pathinfo($file['name'], PATHINFO_EXTENSION) ?: 'png';
        $filename = 'logo_' . time() . '.' . $ext;
        $filepath = $uploadDir . $filename;
        
        if (!move_uploaded_file($file['tmp_name'], $filepath)) {
            Response::error('Error al guardar el archivo. Verifique permisos de ' . $uploadDir);
        }
        
        // Guardar como base64 en BD
        $base64 = 'data:' . $mime . ';base64,' . base64_encode(file_get_contents($filepath));
        $db->prepare("INSERT INTO configuracion (clave, valor, grupo, descripcion, tipo) VALUES ('org_logo_data', ?, 'general', 'Logo base64', 'text') ON DUPLICATE KEY UPDATE valor = ?")
            ->execute([$base64, $base64]);
        
        // URL del endpoint que sirve el logo
        $logoUrl = '/api/v1/configuracion/logo';
        
        $db->prepare("INSERT INTO configuracion (clave, valor, grupo, descripcion, tipo) VALUES ('org_logo', ?, 'general', 'Logo de la organización', 'text') ON DUPLICATE KEY UPDATE valor = ?")
            ->execute([$logoUrl, $logoUrl]);
        
        Response::success([
            'url' => $logoUrl,
            'file' => $filename,
            'path' => $filepath,
            'upload_dir' => $uploadDir,
            'debug' => 'Si no se ve el logo, prueba acceder directamente a: ' . APP_URL . $logoUrl
        ], 'Logo subido correctamente');
    }
    
    /**
     * GET /configuracion/branding - Obtener configuración de marca (público)
     * Devuelve las claves de BD + alias que el sidebar espera
     */
    public static function branding(): void {
        $db = getDB();
        
        try {
            $stmt = $db->query("SELECT clave, valor FROM configuracion WHERE grupo = 'general'");
            $config = [];
            foreach ($stmt->fetchAll() as $row) {
                $config[$row['clave']] = $row['valor'];
            }
            
            // Agregar alias que el sidebar espera
            // BD usa: org_nombre, org_logo → sidebar espera: nombre_empresa, logo_url
            if (!empty($config['org_nombre'])) {
                $config['nombre_empresa'] = $config['org_nombre'];
            }
            if (!empty($config['org_logo'])) {
                $config['logo_url'] = $config['org_logo'];
            }
            
            Response::success($config);
        } catch(Exception $e) {
            Response::success([]);
        }
    }
    
    /**
     * GET /configuracion/logo - Servir el logo directamente (público)
     */
    public static function serveLogo(): void {
        $db = getDB();
        
        try {
            // Primero intentar base64 de BD
            $stmt = $db->prepare("SELECT valor FROM configuracion WHERE clave = 'org_logo_data'");
            $stmt->execute();
            $row = $stmt->fetch();
            
            if ($row && $row['valor']) {
                $parts = explode(',', $row['valor'], 2);
                if (count($parts) === 2) {
                    preg_match('/data:([^;]+);/', $parts[0], $m);
                    $mime = $m[1] ?? 'image/png';
                    header_remove('Content-Type');
                    header('Content-Type: ' . $mime);
                    header('Cache-Control: public, max-age=86400');
                    echo base64_decode($parts[1]);
                    exit;
                }
            }
            
            // Intentar servir desde archivo
            $stmt = $db->prepare("SELECT valor FROM configuracion WHERE clave = 'org_logo'");
            $stmt->execute();
            $row = $stmt->fetch();
            
            if ($row && $row['valor']) {
                $url = $row['valor'];
                $paths = [
                    UPLOAD_DIR . '/branding/' . basename($url),
                    dirname(dirname(__DIR__)) . $url,
                    dirname(dirname(__DIR__)) . '/uploads/branding/' . basename($url),
                    $_SERVER['DOCUMENT_ROOT'] . $url,
                    $_SERVER['DOCUMENT_ROOT'] . '/uploads/branding/' . basename($url),
                ];
                
                foreach ($paths as $p) {
                    if (file_exists($p) && is_file($p)) {
                        $mime = mime_content_type($p) ?: 'image/png';
                        header_remove('Content-Type');
                        header('Content-Type: ' . $mime);
                        header('Cache-Control: public, max-age=86400');
                        readfile($p);
                        exit;
                    }
                }
            }
        } catch(Exception $e) {}
        
        // No logo — return 1x1 transparent pixel
        header_remove('Content-Type');
        header('Content-Type: image/png');
        echo base64_decode('iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mP8/5+hHgAHggJ/PchI7wAAAABJRU5ErkJggg==');
        exit;
    }
    
    /**
     * Asegurar que existan los registros del grupo general
     */
    private static function ensureGeneralConfig(PDO $db): void {
        $defaults = [
            ['org_nombre', 'Nexus IT', 'general', 'Nombre de la organización', 'text'],
            ['org_descripcion', 'Sistema de Gestión de Servicios de TI', 'general', 'Descripción o slogan', 'text'],
            ['org_email', '', 'general', 'Email de contacto', 'email'],
            ['org_telefono', '', 'general', 'Teléfono de contacto', 'text'],
            ['org_direccion', '', 'general', 'Dirección', 'text'],
            ['org_sitio_web', '', 'general', 'Sitio web', 'text'],
            ['org_logo', '', 'general', 'Logo de la organización', 'text'],
            ['brand_color_primary', '#6366f1', 'general', 'Color principal', 'color'],
            ['brand_color_secondary', '#1e293b', 'general', 'Color secundario', 'color'],
            ['brand_color_accent', '#22d3ee', 'general', 'Color de acento', 'color'],
            ['brand_sidebar_bg', '#0f172a', 'general', 'Color fondo sidebar', 'color'],
            ['brand_header_bg', '#1e293b', 'general', 'Color fondo header', 'color'],
        ];
        
        $stmt = $db->prepare("INSERT IGNORE INTO configuracion (clave, valor, grupo, descripcion, tipo) VALUES (?,?,?,?,?)");
        foreach ($defaults as $d) {
            $stmt->execute($d);
        }
    }
}