<?php
class EncuestaController {

    // Crear encuesta y generar token (se llama internamente al cerrar ticket)
    public static function crear(int $ticketId, int $usuarioId): ?string {
        $db = getDB();

        // Verificar si ya existe
        $exists = $db->prepare("SELECT token FROM encuestas_satisfaccion WHERE ticket_id = ?");
        $exists->execute([$ticketId]);
        if ($row = $exists->fetch()) return $row['token'];

        $token = bin2hex(random_bytes(32));
        $db->prepare("INSERT INTO encuestas_satisfaccion (ticket_id, usuario_id, calificacion, token) VALUES (?,?,0,?)")
            ->execute([$ticketId, $usuarioId, $token]);

        return $token;
    }

    // Ver encuesta por token (público, sin auth)
    public static function verPorToken(): void {
        $token = $_GET['token'] ?? '';
        if (empty($token)) Response::error('Token requerido');

        $db = getDB();
        $stmt = $db->prepare("SELECT e.*, t.numero_ticket, t.titulo, t.tecnico_asignado_id,
            u.nombre_completo as usuario_nombre,
            tec.nombre_completo as tecnico_nombre
            FROM encuestas_satisfaccion e
            JOIN tickets t ON e.ticket_id = t.id
            JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN usuarios tec ON t.tecnico_asignado_id = tec.id
            WHERE e.token = ?");
        $stmt->execute([$token]);
        $encuesta = $stmt->fetch();

        if (!$encuesta) Response::notFound('Encuesta no encontrada');
        Response::success($encuesta);
    }

    // Responder encuesta (público, sin auth, por token)
    public static function responder(): void {
        $body = Validator::getBody();
        $token = $body['token'] ?? '';
        if (empty($token)) Response::error('Token requerido');

        $db = getDB();
        $stmt = $db->prepare("SELECT id, ticket_id, respondida_at FROM encuestas_satisfaccion WHERE token = ?");
        $stmt->execute([$token]);
        $encuesta = $stmt->fetch();

        if (!$encuesta) Response::notFound('Encuesta no encontrada');
        if ($encuesta['respondida_at']) Response::error('Esta encuesta ya fue respondida');

        $calificacion = (int)($body['calificacion'] ?? 0);
        if ($calificacion < 1 || $calificacion > 5) Response::error('Calificación debe ser de 1 a 5');

        $facilidad = isset($body['facilidad']) ? max(1, min(5, (int)$body['facilidad'])) : null;
        $tiempo = isset($body['tiempo_respuesta']) ? max(1, min(5, (int)$body['tiempo_respuesta'])) : null;
        $comentario = $body['comentario'] ?? null;

        $db->prepare("UPDATE encuestas_satisfaccion SET calificacion = ?, facilidad = ?, tiempo_respuesta = ?, comentario = ?, respondida_at = NOW() WHERE id = ?")
            ->execute([$calificacion, $facilidad, $tiempo, $comentario, $encuesta['id']]);

        // También actualizar calificación en el ticket
        $db->prepare("UPDATE tickets SET calificacion = ?, comentario_calificacion = ? WHERE id = ?")
            ->execute([$calificacion, $comentario, $encuesta['ticket_id']]);

        Response::success(null, '¡Gracias por tu calificación!');
    }

    // Calificación rápida por URL (clic en estrella del email)
    public static function calificacionRapida(): void {
        $token = $_GET['token'] ?? '';
        $rating = (int)($_GET['r'] ?? 0);

        if (empty($token) || $rating < 1 || $rating > 5) {
            Response::error('Parámetros inválidos');
        }

        $db = getDB();
        $stmt = $db->prepare("SELECT id, ticket_id, respondida_at FROM encuestas_satisfaccion WHERE token = ?");
        $stmt->execute([$token]);
        $encuesta = $stmt->fetch();

        if (!$encuesta) Response::notFound('Encuesta no encontrada');

        // Si no ha sido respondida, registrar calificación rápida
        if (!$encuesta['respondida_at']) {
            $db->prepare("UPDATE encuestas_satisfaccion SET calificacion = ?, respondida_at = NOW() WHERE id = ?")
                ->execute([$rating, $encuesta['id']]);
            $db->prepare("UPDATE tickets SET calificacion = ? WHERE id = ?")
                ->execute([$rating, $encuesta['ticket_id']]);
        }

        // Redirigir a página de encuesta completa
        header("Location: /app/portal/encuesta.html?token={$token}&rated={$rating}");
        exit;
    }

    // Dashboard de encuestas (admin)
    public static function resumen(): void {
        AuthMiddleware::verificar();
        $db = getDB();

        $stats = $db->query("SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN respondida_at IS NOT NULL THEN 1 ELSE 0 END) as respondidas,
            ROUND(AVG(CASE WHEN respondida_at IS NOT NULL THEN calificacion END), 1) as promedio_general,
            ROUND(AVG(CASE WHEN respondida_at IS NOT NULL THEN facilidad END), 1) as promedio_facilidad,
            ROUND(AVG(CASE WHEN respondida_at IS NOT NULL THEN tiempo_respuesta END), 1) as promedio_tiempo,
            SUM(CASE WHEN calificacion = 5 THEN 1 ELSE 0 END) as cinco_estrellas,
            SUM(CASE WHEN calificacion = 4 THEN 1 ELSE 0 END) as cuatro_estrellas,
            SUM(CASE WHEN calificacion = 3 THEN 1 ELSE 0 END) as tres_estrellas,
            SUM(CASE WHEN calificacion = 2 THEN 1 ELSE 0 END) as dos_estrellas,
            SUM(CASE WHEN calificacion = 1 THEN 1 ELSE 0 END) as una_estrella
            FROM encuestas_satisfaccion WHERE respondida_at IS NOT NULL")->fetch();

        $recientes = $db->query("SELECT e.*, t.numero_ticket, t.titulo,
            u.nombre_completo as usuario_nombre,
            tec.nombre_completo as tecnico_nombre
            FROM encuestas_satisfaccion e
            JOIN tickets t ON e.ticket_id = t.id
            JOIN usuarios u ON e.usuario_id = u.id
            LEFT JOIN usuarios tec ON t.tecnico_asignado_id = tec.id
            WHERE e.respondida_at IS NOT NULL
            ORDER BY e.respondida_at DESC LIMIT 20")->fetchAll();

        // Promedio por técnico
        $porTecnico = $db->query("SELECT tec.nombre_completo as tecnico,
            COUNT(*) as encuestas,
            ROUND(AVG(e.calificacion), 1) as promedio
            FROM encuestas_satisfaccion e
            JOIN tickets t ON e.ticket_id = t.id
            JOIN usuarios tec ON t.tecnico_asignado_id = tec.id
            WHERE e.respondida_at IS NOT NULL
            GROUP BY t.tecnico_asignado_id
            ORDER BY promedio DESC")->fetchAll();

        Response::success([
            'stats' => $stats,
            'recientes' => $recientes,
            'por_tecnico' => $porTecnico
        ]);
    }
}
