<?php
class ImportController {

    // GET /importar/plantilla-usuarios — descargar plantilla CSV
    public static function plantillaUsuarios(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="plantilla_usuarios.csv"');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF)); // BOM UTF-8
        fputcsv($out, ['nombre_completo', 'email', 'password', 'rol', 'departamento', 'telefono']);
        fputcsv($out, ['Juan Pérez', 'juan@empresa.com', 'Temporal123', 'usuario', 'Ventas', '+57 300 123 4567']);
        fputcsv($out, ['Ana García', 'ana@empresa.com', 'Temporal456', 'tecnico', 'TI', '+57 301 234 5678']);
        fputcsv($out, ['Carlos López', 'carlos@empresa.com', 'Temporal789', 'usuario', 'RRHH', '']);
        fclose($out);
        exit;
    }

    // POST /importar/usuarios — importar CSV
    public static function importarUsuarios(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);

        if (empty($_FILES['archivo'])) Response::error('Archivo CSV requerido');

        $file = $_FILES['archivo']['tmp_name'];
        $handle = fopen($file, 'r');
        if (!$handle) Response::error('No se pudo leer el archivo');

        // Detectar BOM UTF-8
        $bom = fread($handle, 3);
        if ($bom !== chr(0xEF).chr(0xBB).chr(0xBF)) rewind($handle);

        $header = fgetcsv($handle);
        if (!$header) Response::error('Archivo vacío');

        // Normalizar headers
        $header = array_map(function($h) { return strtolower(trim(str_replace(['"',' '], ['','_'], $h))); }, $header);

        $required = ['nombre_completo', 'email', 'password'];
        foreach ($required as $r) {
            if (!in_array($r, $header)) Response::error("Columna requerida faltante: $r");
        }

        $db = getDB();
        $creados = 0; $errores = []; $fila = 1;
        $rolesValidos = ['admin','supervisor','tecnico','tecnico_n2','auditor','usuario'];

        while (($row = fgetcsv($handle)) !== false) {
            $fila++;
            if (count($row) < count($header)) { $errores[] = "Fila $fila: columnas insuficientes"; continue; }

            $data = array_combine($header, $row);
            $nombre = trim($data['nombre_completo'] ?? '');
            $email = trim($data['email'] ?? '');
            $pass = trim($data['password'] ?? '');
            $rol = strtolower(trim($data['rol'] ?? 'usuario'));
            $depto = trim($data['departamento'] ?? '') ?: null;
            $tel = trim($data['telefono'] ?? '') ?: null;

            if (empty($nombre) || empty($email)) { $errores[] = "Fila $fila: nombre y email obligatorios"; continue; }
            if (!filter_var($email, FILTER_VALIDATE_EMAIL)) { $errores[] = "Fila $fila: email inválido ($email)"; continue; }
            if (strlen($pass) < 6) { $errores[] = "Fila $fila: contraseña mínimo 6 caracteres"; continue; }
            if (!in_array($rol, $rolesValidos)) { $rol = 'usuario'; }

            // Verificar duplicado
            $exists = $db->prepare("SELECT id FROM usuarios WHERE email = ?");
            $exists->execute([$email]);
            if ($exists->fetch()) { $errores[] = "Fila $fila: email ya existe ($email)"; continue; }

            try {
                $hash = password_hash($pass, PASSWORD_BCRYPT);
                $db->prepare("INSERT INTO usuarios (uuid, nombre_completo, email, password_hash, rol, departamento, telefono, activo) VALUES (UUID(),?,?,?,?,?,?,1)")
                    ->execute([$nombre, $email, $hash, $rol, $depto, $tel]);
                $creados++;

                // Log
                $userId = (int)$db->lastInsertId();
                try { $db->prepare("INSERT INTO log_actividad (usuario_id, accion, entidad_tipo, entidad_id, ip_address, descripcion) VALUES (?,?,?,?,?,?)")
                    ->execute([$payload['sub'], 'importar_usuario', 'usuarios', $userId, $_SERVER['REMOTE_ADDR'] ?? null, "Importación masiva: $nombre ($email)"]); } catch (\Exception $e) {}

            } catch (\Exception $e) {
                $errores[] = "Fila $fila: error BD — " . $e->getMessage();
            }
        }
        fclose($handle);

        Response::success([
            'creados' => $creados,
            'errores' => count($errores),
            'detalle_errores' => $errores
        ], "$creados usuario(s) importado(s)");
    }

    // GET /importar/plantilla-activos — plantilla para activos
    public static function plantillaActivos(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="plantilla_activos.csv"');
        $out = fopen('php://output', 'w');
        fprintf($out, chr(0xEF).chr(0xBB).chr(0xBF));
        fputcsv($out, ['nombre', 'tipo', 'marca', 'modelo', 'numero_serie', 'codigo_interno', 'sistema_operativo', 'estado', 'ubicacion', 'costo_adquisicion', 'fecha_compra', 'garantia_hasta', 'email_usuario_asignado']);
        fputcsv($out, ['Laptop Juan', 'laptop', 'Dell', 'Latitude 5540', 'SN123456', 'LAP-0010', 'Windows 11 Pro', 'activo', 'Oficina Bogotá', '3500000', '2024-01-15', '2027-01-15', 'juan@empresa.com']);
        fclose($out);
        exit;
    }
}
