<?php
/**
 * NEXUS IT - IntegracionController
 * API pública, webhooks, API keys para integraciones externas
 */
class IntegracionController {

    /**
     * POST /integracion/api-key — Generar API key para integraciones
     */
    public static function generarApiKey(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $body = Validator::getBody();
        $nombre = $body['nombre'] ?? 'API Key';

        $apiKey = 'nxk_' . bin2hex(random_bytes(24));
        $db = getDB();

        // Store in configuracion table
        $db->prepare("INSERT INTO configuracion (clave, valor, grupo, tipo, descripcion) VALUES (?,?,?,?,?) ON DUPLICATE KEY UPDATE valor = ?, updated_at = NOW()")
            ->execute([
                'api_key_' . substr(md5($apiKey), 0, 8),
                json_encode(['key' => password_hash($apiKey, PASSWORD_BCRYPT), 'nombre' => $nombre, 'creado' => date('Y-m-d H:i:s'), 'creado_por' => $payload['sub']]),
                'integracion', 'json', "API Key: $nombre",
                json_encode(['key' => password_hash($apiKey, PASSWORD_BCRYPT), 'nombre' => $nombre, 'creado' => date('Y-m-d H:i:s'), 'creado_por' => $payload['sub']])
            ]);

        AuditService::log('crear_api_key', 'configuracion', null, null, ['nombre' => $nombre]);
        Response::created(['api_key' => $apiKey, 'nombre' => $nombre], 'API Key generada. Guárdala, no se mostrará de nuevo.');
    }

    /**
     * GET /integracion/api-keys — Listar API keys
     */
    public static function listarApiKeys(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $db = getDB();
        $stmt = $db->query("SELECT clave, valor, updated_at FROM configuracion WHERE grupo = 'integracion' AND clave LIKE 'api_key_%'");
        $keys = [];
        foreach ($stmt->fetchAll() as $row) {
            $data = json_decode($row['valor'], true);
            $keys[] = ['id' => $row['clave'], 'nombre' => $data['nombre'] ?? '—', 'creado' => $data['creado'] ?? $row['updated_at']];
        }
        Response::success($keys);
    }

    /**
     * DELETE /integracion/api-key/:clave — Revocar API key
     */
    public static function revocarApiKey(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireAdmin($payload);
        $body = Validator::getBody();
        if (empty($body['clave'])) Response::error('Clave requerida');
        $db = getDB();
        $db->prepare("DELETE FROM configuracion WHERE clave = ? AND grupo = 'integracion'")->execute([$body['clave']]);
        AuditService::log('revocar_api_key', 'configuracion', null, ['clave' => $body['clave']]);
        Response::success(null, 'API Key revocada');
    }

    /**
     * POST /integracion/webhook/ticket — Crear ticket via webhook (API key auth)
     */
    public static function webhookCrearTicket(): void {
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? $_GET['api_key'] ?? '';
        if (!self::validarApiKey($apiKey)) Response::forbidden('API Key inválida');

        $body = Validator::getBody();
        if (empty($body['titulo'])) Response::error('titulo es requerido');

        $db = getDB();

        // Find or create solicitante
        $solicitanteId = 1; // Default admin
        if (!empty($body['email_solicitante'])) {
            $stmt = $db->prepare("SELECT id FROM usuarios WHERE email = ?");
            $stmt->execute([$body['email_solicitante']]);
            $user = $stmt->fetch();
            if ($user) $solicitanteId = $user['id'];
        }

        $year = date('y'); $month = date('m');
        $seq = (int)$db->query("SELECT COUNT(*) FROM tickets WHERE YEAR(created_at)=YEAR(NOW()) AND MONTH(created_at)=MONTH(NOW())")->fetchColumn() + 1;
        $numero = 'TK' . $year . $month . str_pad($seq, 4, '0', STR_PAD_LEFT);

        $canal = $body['canal_origen'] ?? 'email';
        $stmt = $db->prepare("INSERT INTO tickets (numero_ticket, titulo, descripcion, solicitante_id, prioridad, estado, canal_origen) VALUES (?,?,?,?,?,?,?)");
        $stmt->execute([$numero, $body['titulo'], $body['descripcion'] ?? null, $solicitanteId, $body['prioridad'] ?? 'media', 'abierto', $canal]);
        $id = (int)$db->lastInsertId();

        AuditService::log('webhook_crear_ticket', 'tickets', $id, null, ['canal' => $canal]);
        Response::created(['id' => $id, 'numero_ticket' => $numero]);
    }

    /**
     * GET /integracion/stats — Stats públicas (con API key)
     */
    public static function statsPublicas(): void {
        $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? $_GET['api_key'] ?? '';
        if (!self::validarApiKey($apiKey)) Response::forbidden('API Key inválida');

        $db = getDB();
        Response::success([
            'tickets_abiertos' => (int)$db->query("SELECT COUNT(*) FROM tickets WHERE estado IN ('abierto','en_progreso','pendiente')")->fetchColumn(),
            'tickets_hoy' => (int)$db->query("SELECT COUNT(*) FROM tickets WHERE DATE(created_at) = CURDATE()")->fetchColumn(),
            'total_activos' => (int)$db->query("SELECT COUNT(*) FROM activos")->fetchColumn(),
            'alertas_activas' => (int)$db->query("SELECT COUNT(*) FROM alertas WHERE resuelta = 0")->fetchColumn()
        ]);
    }

    private static function validarApiKey(string $key): bool {
        if (empty($key) || !str_starts_with($key, 'nxk_')) return false;
        $db = getDB();
        $stmt = $db->query("SELECT valor FROM configuracion WHERE grupo = 'integracion' AND clave LIKE 'api_key_%'");
        foreach ($stmt->fetchAll() as $row) {
            $data = json_decode($row['valor'], true);
            if (!empty($data['key']) && password_verify($key, $data['key'])) return true;
        }
        return false;
    }
}
