<?php
/**
 * NEXUS IT - Knowledge Base Controller
 * Artículos de autoservicio
 */

class KnowledgeBaseController {
    
    /**
     * GET /kb/articulos - Listar artículos
     */
    public static function index(): void {
        $auth = AuthMiddleware::verificar();
        $db = getDB();
        
        $page    = max(1, intval($_GET['page'] ?? 1));
        $perPage = min(MAX_PAGE_SIZE, max(1, intval($_GET['per_page'] ?? DEFAULT_PAGE_SIZE)));
        $offset  = ($page - 1) * $perPage;
        
        $where = ['a.publicado = 1'];
        $params = [];
        
        // Técnicos/admins ven borradores también
        if (in_array($auth['rol'], ['admin', 'tecnico']) && !empty($_GET['incluir_borradores'])) {
            $where = [];
        }
        
        if (!empty($_GET['categoria_id'])) {
            $where[] = 'a.categoria_id = ?';
            $params[] = intval($_GET['categoria_id']);
        }
        
        $whereSQL = $where ? 'WHERE ' . implode(' AND ', $where) : '';
        
        $stmt = $db->prepare("SELECT COUNT(*) FROM articulos_kb a $whereSQL");
        $stmt->execute($params);
        $total = (int)$stmt->fetchColumn();
        
        $sql = "
            SELECT a.id, a.titulo, a.slug, a.tags, a.vistas, a.util_si, a.util_no, 
                a.publicado, a.created_at, a.updated_at,
                u.nombre_completo AS autor_nombre,
                c.nombre AS categoria_nombre
            FROM articulos_kb a
            LEFT JOIN usuarios u ON a.autor_id = u.id
            LEFT JOIN categorias c ON a.categoria_id = c.id
            $whereSQL
            ORDER BY a.vistas DESC, a.created_at DESC
            LIMIT $perPage OFFSET $offset
        ";
        
        $stmt = $db->prepare($sql);
        $stmt->execute($params);
        
        Response::paginated($stmt->fetchAll(), $total, $page, $perPage);
    }
    
    /**
     * GET /kb/articulos/{slug} - Ver artículo por slug
     */
    public static function ver(string $slug): void {
        $auth = AuthMiddleware::verificar();
        $db = getDB();
        
        $stmt = $db->prepare("
            SELECT a.*, u.nombre_completo AS autor_nombre, c.nombre AS categoria_nombre
            FROM articulos_kb a
            LEFT JOIN usuarios u ON a.autor_id = u.id
            LEFT JOIN categorias c ON a.categoria_id = c.id
            WHERE a.slug = ?
        ");
        $stmt->execute([$slug]);
        $articulo = $stmt->fetch();
        
        if (!$articulo) Response::notFound('Artículo no encontrado');
        
        if (!$articulo['publicado'] && $auth['rol'] === 'usuario') {
            Response::notFound('Artículo no encontrado');
        }
        
        // Incrementar vistas
        $db->prepare("UPDATE articulos_kb SET vistas = vistas + 1 WHERE id = ?")->execute([$articulo['id']]);
        
        Response::success($articulo);
    }
    
    /**
     * GET /kb/buscar?q= - Búsqueda fulltext
     */
    public static function buscar(): void {
        $auth = AuthMiddleware::verificar();
        $db = getDB();
        
        $q = $_GET['q'] ?? '';
        if (strlen(trim($q)) < 2) Response::error('Búsqueda muy corta (mínimo 2 caracteres)');
        
        $stmt = $db->prepare("
            SELECT a.id, a.titulo, a.slug, a.tags, a.vistas,
                SUBSTRING(a.contenido, 1, 200) AS extracto,
                u.nombre_completo AS autor_nombre,
                MATCH(a.titulo, a.contenido, a.tags) AGAINST(? IN NATURAL LANGUAGE MODE) AS relevancia
            FROM articulos_kb a
            LEFT JOIN usuarios u ON a.autor_id = u.id
            WHERE a.publicado = 1 AND MATCH(a.titulo, a.contenido, a.tags) AGAINST(? IN NATURAL LANGUAGE MODE)
            ORDER BY relevancia DESC
            LIMIT 20
        ");
        $stmt->execute([$q, $q]);
        
        Response::success($stmt->fetchAll());
    }
    
    /**
     * POST /kb/articulos - Crear artículo (técnico/admin)
     */
    public static function crear(): void {
        $auth = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($auth);
        
        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        
        $v = new Validator($data);
        $v->required('titulo', 'Título')
          ->maxLength('titulo', 255)
          ->required('contenido', 'Contenido');
        $v->validate();
        
        $slug = self::generarSlug($data['titulo']);
        
        $db = getDB();
        
        // Verificar slug único
        $stmt = $db->prepare("SELECT COUNT(*) FROM articulos_kb WHERE slug = ?");
        $stmt->execute([$slug]);
        if ($stmt->fetchColumn() > 0) {
            $slug .= '-' . substr(uniqid(), -4);
        }
        
        $stmt = $db->prepare("
            INSERT INTO articulos_kb (autor_id, categoria_id, titulo, contenido, slug, tags, publicado)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $auth['sub'],
            $data['categoria_id'] ?? null,
            trim($data['titulo']),
            $data['contenido'],
            $slug,
            $data['tags'] ?? null,
            isset($data['publicado']) ? intval($data['publicado']) : 0
        ]);
        
        $id = $db->lastInsertId();
        
        $stmt = $db->prepare("SELECT * FROM articulos_kb WHERE id = ?");
        $stmt->execute([$id]);
        
        Response::created($stmt->fetch(), 'Artículo creado');
    }
    
    /**
     * PUT /kb/articulos/{id} - Editar artículo
     */
    public static function actualizar(int $id): void {
        $auth = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($auth);
        
        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        $db = getDB();
        
        $stmt = $db->prepare("SELECT id FROM articulos_kb WHERE id = ?");
        $stmt->execute([$id]);
        if (!$stmt->fetch()) Response::notFound('Artículo no encontrado');
        
        $campos = [];
        $valores = [];
        
        $editables = ['titulo', 'contenido', 'tags', 'publicado', 'categoria_id'];
        foreach ($editables as $c) {
            if (array_key_exists($c, $data)) {
                $campos[] = "`$c` = ?";
                $valores[] = $data[$c];
            }
        }
        
        if (empty($campos)) Response::error('Sin campos para actualizar');
        
        $valores[] = $id;
        $db->prepare("UPDATE articulos_kb SET " . implode(', ', $campos) . " WHERE id = ?")->execute($valores);
        
        Response::success(null, 'Artículo actualizado');
    }
    
    /**
     * POST /kb/articulos/{id}/util - Votar utilidad
     */
    public static function votar(int $id): void {
        $auth = AuthMiddleware::verificar();
        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        $db = getDB();
        
        $campo = !empty($data['util']) ? 'util_si' : 'util_no';
        $db->prepare("UPDATE articulos_kb SET $campo = $campo + 1 WHERE id = ?")->execute([$id]);
        
        Response::success(null, 'Voto registrado');
    }
    
    // === HELPERS ===
    
    private static function generarSlug(string $texto): string {
        $slug = strtolower(trim($texto));
        $slug = preg_replace('/[áàäâ]/u', 'a', $slug);
        $slug = preg_replace('/[éèëê]/u', 'e', $slug);
        $slug = preg_replace('/[íìïî]/u', 'i', $slug);
        $slug = preg_replace('/[óòöô]/u', 'o', $slug);
        $slug = preg_replace('/[úùüû]/u', 'u', $slug);
        $slug = preg_replace('/ñ/u', 'n', $slug);
        $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
        $slug = trim($slug, '-');
        return $slug ?: 'articulo-' . uniqid();
    }
}
