¿<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Nexus IT — Reportes & Auditoría</title>
<link href="https://fonts.googleapis.com/css2?family=DM+Sans:wght@300;400;500;600;700;800&family=JetBrains+Mono:wght@400;600&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/css/app.css">
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<style>
.kpi-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 1rem; margin-bottom: 1.5rem; }
.kpi-card { background: var(--bg-card); border: 1px solid var(--border-color); border-radius: 12px; padding: 1.2rem; }
.kpi-label { font-size: 0.75rem; color: var(--text-muted); text-transform: uppercase; letter-spacing: 0.5px; margin-bottom: 0.4rem; }
.kpi-value { font-size: 1.8rem; font-weight: 700; color: var(--text-primary); }
.kpi-sub { font-size: 0.75rem; color: var(--text-muted); margin-top: 0.25rem; }
.kpi-value.green { color: #22c55e; }
.kpi-value.yellow { color: #eab308; }
.kpi-value.red { color: #ef4444; }
.kpi-value.blue { color: #3b82f6; }

.charts-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(420px, 1fr)); gap: 1.5rem; margin-bottom: 1.5rem; }
.chart-card { background: var(--bg-card); border: 1px solid var(--border-color); border-radius: 12px; padding: 1.5rem; }
.chart-card h3 { font-size: 0.95rem; font-weight: 600; color: var(--text-primary); margin-bottom: 1rem; }
.chart-card canvas { max-height: 300px; }

.full-width { grid-column: 1 / -1; }

.date-range { display: flex; align-items: center; gap: 0.75rem; }
.date-range input[type="date"] { background: var(--bg-card); border: 1px solid var(--border-color); color: var(--text-primary); padding: 0.4rem 0.6rem; border-radius: 8px; font-family: inherit; font-size: 0.85rem; }

.tech-table { width: 100%; border-collapse: collapse; margin-top: 0.5rem; }
.tech-table th { font-size: 0.7rem; text-transform: uppercase; letter-spacing: 0.5px; color: var(--text-muted); text-align: left; padding: 0.5rem; border-bottom: 1px solid var(--border-color); }
.tech-table td { padding: 0.6rem 0.5rem; font-size: 0.85rem; border-bottom: 1px solid var(--border-color, rgba(255,255,255,0.06)); }
.tech-table tr:last-child td { border-bottom: none; }
.bar-bg { background: rgba(255,255,255,0.06); border-radius: 4px; height: 6px; width: 100%; position: relative; }
.bar-fill { position: absolute; top: 0; left: 0; height: 100%; border-radius: 4px; }
.star { color: #eab308; }

.export-bar { display: flex; gap: 0.5rem; }
.tab-group { display: flex; gap: 0.25rem; background: rgba(255,255,255,0.04); border-radius: 8px; padding: 3px; margin-bottom: 1rem; }
.tab-btn { padding: 0.4rem 1rem; border-radius: 6px; font-size: 0.8rem; cursor: pointer; border: none; background: transparent; color: var(--text-muted); font-family: inherit; }
.tab-btn.active { background: var(--primary); color: #fff; }

.sla-badge { display: inline-block; padding: 2px 8px; border-radius: 6px; font-size: 0.75rem; font-weight: 600; }
.sla-ok { background: rgba(34,197,94,0.15); color: #22c55e; }
.sla-warn { background: rgba(234,179,8,0.15); color: #eab308; }
.sla-bad { background: rgba(239,68,68,0.15); color: #ef4444; }
</style>
</head>
<body>
<div class="app-layout">
  <aside class="sidebar" id="sidebar"></aside>

  <main class="main-content">
    <header class="top-bar">
      <div class="top-bar-left">
        <button class="btn-menu-mobile" onclick="document.getElementById('sidebar').classList.toggle('open')"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round"><line x1="4" y1="12" x2="20" y2="12"/><line x1="4" y1="6" x2="20" y2="6"/><line x1="4" y1="18" x2="20" y2="18"/></svg></button>
        <h1 class="page-title">Reportes & Auditoría</h1>
      </div>
      <div class="top-bar-right">
        <div class="date-range">
          <input type="date" id="fechaDesde">
          <span style="color:var(--text-muted)">a</span>
          <input type="date" id="fechaHasta">
          <button class="btn btn-primary btn-sm" onclick="cargarTodo()">Aplicar</button>
        </div>
        <div class="export-bar">
          <button class="btn btn-sm" onclick="exportarCSV()" title="Exportar tickets"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"/><polyline points="7,10 12,15 17,10"/><line x1="12" y1="15" x2="12" y2="3"/></svg> CSV</button>
        </div>
      </div>
    </header>

    <div class="content-area">
      <!-- TAB NAVIGATION -->
      <div class="tab-group">
        <button class="tab-btn active" data-tab="general" onclick="switchTab('general')">General</button>
        <button class="tab-btn" data-tab="usuarios" onclick="switchTab('usuarios')">Por Usuario</button>
        <button class="tab-btn" data-tab="tiempos" onclick="switchTab('tiempos')">Tiempos</button>
        <button class="tab-btn" data-tab="sla" onclick="switchTab('sla')">SLA</button>
        <button class="tab-btn" data-tab="tecnicos" onclick="switchTab('tecnicos')">Técnicos</button>
        <button class="tab-btn" data-tab="activos" onclick="switchTab('activos')">Activos</button>
      </div>

      <!-- TAB: GENERAL -->
      <div id="tab-general" class="tab-content">
        <div class="kpi-grid" id="kpiGrid"></div>
        <div class="charts-grid">
          <div class="chart-card full-width">
            <h3>Tendencia de Tickets (últimos 30 días)</h3>
            <canvas id="chartTendencia"></canvas>
          </div>
          <div class="chart-card">
            <h3>Tickets por Categoría</h3>
            <canvas id="chartCategorias"></canvas>
          </div>
          <div class="chart-card">
            <h3>Distribución por Estado</h3>
            <canvas id="chartEstados"></canvas>
          </div>
        </div>
      </div>

      <!-- TAB: POR USUARIO (Solicitantes) -->
      <div id="tab-usuarios" class="tab-content" style="display:none">
        <div class="chart-card">
          <h3>Tickets Creados por Usuario</h3>
          <p style="color:var(--text-muted);font-size:0.82rem;margin-bottom:1rem">Quién abrió más casos en el período seleccionado y tiempos promedio de atención</p>
          <table class="tech-table">
            <thead>
              <tr>
                <th>Usuario</th>
                <th>Departamento</th>
                <th>Total</th>
                <th>Abiertos</th>
                <th>Resueltos</th>
                <th>Tiempo Prom.</th>
                <th>Satisf.</th>
                <th>Distribución</th>
              </tr>
            </thead>
            <tbody id="usuariosTabla">
              <tr><td colspan="8" style="text-align:center;padding:40px;color:var(--text-muted)">Cargando...</td></tr>
            </tbody>
          </table>
        </div>
        <div class="charts-grid" style="margin-top:1.5rem">
          <div class="chart-card">
            <h3>Top Solicitantes</h3>
            <canvas id="chartUsuarios"></canvas>
          </div>
          <div class="chart-card">
            <h3>Tickets por Departamento</h3>
            <canvas id="chartDeptos"></canvas>
          </div>
        </div>
      </div>

      <!-- TAB: TIEMPOS DE ATENCIÓN -->
      <div id="tab-tiempos" class="tab-content" style="display:none">
        <div class="kpi-grid">
          <div class="kpi-card"><div class="kpi-label">Primera Respuesta</div><div class="kpi-value blue" id="kpiPrimeraResp">—</div><div class="kpi-sub">promedio en horas</div></div>
          <div class="kpi-card"><div class="kpi-label">Resolución Promedio</div><div class="kpi-value green" id="kpiResolucion">—</div><div class="kpi-sub">horas</div></div>
          <div class="kpi-card"><div class="kpi-label">Resueltos &lt;24h</div><div class="kpi-value green" id="kpiMenos24">0</div><div class="kpi-sub" id="kpiMenos24Pct">0%</div></div>
          <div class="kpi-card"><div class="kpi-label">Resueltos &gt;48h</div><div class="kpi-value red" id="kpiMas48">0</div><div class="kpi-sub" id="kpiMas48Pct">0%</div></div>
        </div>
        <div class="chart-card" style="margin-top:1rem">
          <h3>Detalle de Tiempos por Ticket</h3>
          <table class="tech-table">
            <thead>
              <tr>
                <th>Ticket</th>
                <th>Título</th>
                <th>Prioridad</th>
                <th>Técnico</th>
                <th>Tiempo Total</th>
                <th>1ra Respuesta</th>
              </tr>
            </thead>
            <tbody id="tiemposTabla">
              <tr><td colspan="6" style="text-align:center;padding:40px;color:var(--text-muted)">Cargando...</td></tr>
            </tbody>
          </table>
        </div>
      </div>

      <!-- TAB: SLA -->
      <div id="tab-sla" class="tab-content" style="display:none">
        <div class="kpi-grid" id="slaKpis"></div>
        <div class="charts-grid">
          <div class="chart-card full-width">
            <h3>Cumplimiento SLA por Categoría</h3>
            <canvas id="chartSLA"></canvas>
          </div>
        </div>
        <div class="chart-card" style="margin-top:1rem">
          <h3>Tickets Vencidos Abiertos</h3>
          <div id="slaTabla"></div>
        </div>
      </div>

      <!-- TAB: TECNICOS -->
      <div id="tab-tecnicos" class="tab-content" style="display:none">
        <div class="charts-grid">
          <div class="chart-card full-width">
            <h3>Rendimiento de Técnicos</h3>
            <div id="techTabla"></div>
          </div>
          <div class="chart-card">
            <h3>Tickets por Técnico</h3>
            <canvas id="chartTecnicos"></canvas>
          </div>
          <div class="chart-card">
            <h3>Satisfacción por Técnico</h3>
            <canvas id="chartSatisfaccion"></canvas>
          </div>
        </div>
      </div>

      <!-- TAB: ACTIVOS -->
      <div id="tab-activos" class="tab-content" style="display:none">
        <div class="kpi-grid" id="activoKpis"></div>
        <div class="charts-grid">
          <div class="chart-card">
            <h3> Activos por Tipo</h3>
            <canvas id="chartActivosTipo"></canvas>
          </div>
          <div class="chart-card">
            <h3> Activos por Estado</h3>
            <canvas id="chartActivosEstado"></canvas>
          </div>
          <div class="chart-card">
            <h3> Activos por Departamento</h3>
            <canvas id="chartActivosDepto"></canvas>
          </div>
        </div>
      </div>

    </div>
  </main>
</div>

<script src="../assets/js/api.js"></script>
<script src="../assets/js/auth.js"></script>
<script src="../assets/js/sidebar.js"></script>
<script src="../assets/js/notifications.js"></script>
<script src="../assets/js/utils.js"></script>
<script>
let charts = {};

if (!NexusAuth.requireAuth(['admin','tecnico','supervisor'])) throw 'No auth';

// Colores del design system
const COLORS = ['#3b82f6','#22c55e','#ef4444','#eab308','#8b5cf6','#ec4899','#f97316','#06b6d4','#64748b','#10b981'];
const CHART_DEFAULTS = {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
        legend: { labels: { color: '#94a3b8', font: { family: 'Outfit', size: 12 } } },
        tooltip: { backgroundColor: '#1e293b', titleColor: '#e2e8f0', bodyColor: '#94a3b8', borderColor: '#334155', borderWidth: 1 }
    },
    scales: {
        x: { ticks: { color: '#64748b', font: { size: 11 } }, grid: { color: 'rgba(255,255,255,0.04)' } },
        y: { ticks: { color: '#64748b', font: { size: 11 } }, grid: { color: 'rgba(255,255,255,0.06)' }, beginAtZero: true }
    }
};

// Init
document.addEventListener('DOMContentLoaded', () => {
    const hoy = new Date();
    const inicio = new Date(hoy.getFullYear(), hoy.getMonth(), 1);
    document.getElementById('fechaDesde').value = inicio.toISOString().split('T')[0];
    document.getElementById('fechaHasta').value = hoy.toISOString().split('T')[0];
    Chart.defaults.font.family = 'Outfit';
    cargarTodo();
});

function switchTab(tab) {
    document.querySelectorAll('.tab-content').forEach(t => t.style.display = 'none');
    document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
    document.getElementById('tab-' + tab).style.display = 'block';
    document.querySelector(`[data-tab="${tab}"]`).classList.add('active');
    // Lazy load data per tab
    if (tab === 'usuarios') cargarUsuarios();
    if (tab === 'tiempos') cargarTiempos();
    if (tab === 'sla') cargarSLA();
    if (tab === 'tecnicos') cargarTecnicos();
    if (tab === 'activos') cargarActivos();
}

async function api(url) { const d = await NexusAPI.get(url); if(!d.success) throw new Error(d.message); return d.data; }

function destroyChart(id) {
    if (charts[id]) { charts[id].destroy(); delete charts[id]; }
}

async function cargarTodo() {
    const desde = document.getElementById('fechaDesde').value;
    const hasta = document.getElementById('fechaHasta').value;
    try {
        await Promise.all([
            cargarKPIs(desde, hasta),
            cargarTendencia(),
            cargarCategorias(),
            cargarEstados()
        ]);
        // Reload active tab
        const active = document.querySelector('.tab-btn.active');
        if (active) switchTab(active.dataset.tab);
    } catch(e) {
        console.error('Error cargando reportes:', e);
    }
}

// ========== GENERAL ==========
async function cargarKPIs(desde, hasta) {
    const d = await api(`/reportes/resumen?desde=${desde}&hasta=${hasta}`);
    const slaColor = d.sla_cumplimiento_pct >= 90 ? 'green' : d.sla_cumplimiento_pct >= 70 ? 'yellow' : 'red';
    document.getElementById('kpiGrid').innerHTML = `
        <div class="kpi-card"><div class="kpi-label">Tickets Creados</div><div class="kpi-value blue">${d.tickets_creados}</div><div class="kpi-sub">En el período</div></div>
        <div class="kpi-card"><div class="kpi-label">Tickets Resueltos</div><div class="kpi-value green">${d.tickets_resueltos}</div><div class="kpi-sub">En el período</div></div>
        <div class="kpi-card"><div class="kpi-label">Pendientes</div><div class="kpi-value yellow">${d.tickets_pendientes}</div><div class="kpi-sub">Total activos</div></div>
        <div class="kpi-card"><div class="kpi-label">Tiempo Resolución</div><div class="kpi-value">${d.promedio_resolucion_h || '—'}h</div><div class="kpi-sub">Promedio</div></div>
        <div class="kpi-card"><div class="kpi-label">SLA Cumplimiento</div><div class="kpi-value ${slaColor}">${d.sla_cumplimiento_pct}%</div><div class="kpi-sub">Resueltos a tiempo</div></div>
        <div class="kpi-card"><div class="kpi-label">1ra Respuesta</div><div class="kpi-value">${d.promedio_primera_respuesta_h || '—'}h</div><div class="kpi-sub">Promedio</div></div>
        <div class="kpi-card"><div class="kpi-label">Satisfacción</div><div class="kpi-value">${d.satisfaccion_promedio ? d.satisfaccion_promedio + '/5' : '—'}</div><div class="kpi-sub">${d.total_calificaciones} calificaciones</div></div>
        <div class="kpi-card"><div class="kpi-label">Alertas Activas</div><div class="kpi-value ${d.alertas_activas > 0 ? 'red' : 'green'}">${d.alertas_activas}</div><div class="kpi-sub">${d.mantenimientos_pendientes} mant. pendientes</div></div>
    `;
}

async function cargarTendencia() {
    const d = await api('/reportes/tickets-por-dia?dias=30');
    destroyChart('tendencia');
    charts.tendencia = new Chart(document.getElementById('chartTendencia'), {
        type: 'line',
        data: {
            labels: d.map(r => { const dt = new Date(r.fecha); return dt.getDate() + '/' + (dt.getMonth()+1); }),
            datasets: [{
                label: 'Creados',
                data: d.map(r => r.creados),
                borderColor: '#3b82f6',
                backgroundColor: 'rgba(59,130,246,0.1)',
                fill: true,
                tension: 0.3,
                pointRadius: 3
            }, {
                label: 'Resueltos mismo día',
                data: d.map(r => r.resueltos_mismo_dia),
                borderColor: '#22c55e',
                backgroundColor: 'rgba(34,197,94,0.1)',
                fill: true,
                tension: 0.3,
                pointRadius: 3
            }]
        },
        options: { ...CHART_DEFAULTS, plugins: { ...CHART_DEFAULTS.plugins } }
    });
}

async function cargarCategorias() {
    const d = await api('/reportes/tickets-por-categoria');
    destroyChart('categorias');
    charts.categorias = new Chart(document.getElementById('chartCategorias'), {
        type: 'doughnut',
        data: {
            labels: d.map(r => r.nombre || 'Sin categoría'),
            datasets: [{ data: d.map(r => r.total), backgroundColor: d.map((r,i) => r.color || COLORS[i % COLORS.length]) }]
        },
        options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { position: 'right', labels: { color: '#94a3b8', font: { size: 11 } } } }, cutout: '55%' }
    });
}

async function cargarEstados() {
    const d = await api('/dashboard/tickets-por-estado');
    const coloresEstado = { abierto: '#3b82f6', en_progreso: '#eab308', pendiente: '#8b5cf6', resuelto: '#22c55e', cerrado: '#64748b' };
    destroyChart('estados');
    charts.estados = new Chart(document.getElementById('chartEstados'), {
        type: 'doughnut',
        data: {
            labels: d.map(r => r.estado.replace('_', ' ')),
            datasets: [{ data: d.map(r => r.total), backgroundColor: d.map(r => coloresEstado[r.estado] || '#64748b') }]
        },
        options: { responsive: true, maintainAspectRatio: false, plugins: { legend: { position: 'right', labels: { color: '#94a3b8', font: { size: 11 } } } }, cutout: '55%' }
    });
}

// ========== POR USUARIO ==========
async function cargarUsuarios() {
    const desde = document.getElementById('fechaDesde').value;
    const hasta = document.getElementById('fechaHasta').value;
    const d = await api('/reportes/tickets-por-usuario?desde=' + desde + '&hasta=' + hasta);
    const tbody = document.getElementById('usuariosTabla');
    
    if (!d || d.length === 0) {
        tbody.innerHTML = '<tr><td colspan="8" style="text-align:center;padding:40px;color:var(--text-muted)">No hay datos en este período</td></tr>';
        return;
    }
    
    const maxTickets = Math.max(...d.map(u => u.total_tickets));
    
    tbody.innerHTML = d.map(u => {
        const pct = maxTickets > 0 ? Math.round((u.total_tickets / maxTickets) * 100) : 0;
        const sat = u.satisfaccion_promedio ? '<span class="star">★</span> ' + u.satisfaccion_promedio : '—';
        return `<tr>
            <td><strong>${u.nombre_completo}</strong><br><span style="font-size:11px;color:var(--text-muted)">${u.email}</span></td>
            <td>${u.departamento || '—'}</td>
            <td><strong>${u.total_tickets}</strong></td>
            <td style="color:#eab308">${u.tickets_abiertos || 0}</td>
            <td style="color:#22c55e">${u.tickets_resueltos || 0}</td>
            <td>${u.tiempo_promedio_h ? u.tiempo_promedio_h + 'h' : '—'}</td>
            <td>${sat}</td>
            <td style="width:120px"><div class="bar-bg"><div class="bar-fill" style="width:${pct}%;background:#3b82f6"></div></div></td>
        </tr>`;
    }).join('');

    // Chart top usuarios
    destroyChart('usuarios');
    const top = d.slice(0, 10);
    charts.usuarios = new Chart(document.getElementById('chartUsuarios'), {
        type: 'bar',
        data: {
            labels: top.map(u => u.nombre_completo.split(' ')[0]),
            datasets: [{ label: 'Tickets', data: top.map(u => u.total_tickets), backgroundColor: '#3b82f6', borderRadius: 6 }]
        },
        options: { ...CHART_DEFAULTS, indexAxis: 'y', plugins: { ...CHART_DEFAULTS.plugins, legend: { display: false } } }
    });

    // Chart por departamento
    destroyChart('deptos');
    const deptos = {};
    d.forEach(u => { const dep = u.departamento || 'Sin asignar'; deptos[dep] = (deptos[dep] || 0) + u.total_tickets; });
    const deptEntries = Object.entries(deptos).sort((a, b) => b[1] - a[1]);
    charts.deptos = new Chart(document.getElementById('chartDeptos'), {
        type: 'doughnut',
        data: {
            labels: deptEntries.map(e => e[0]),
            datasets: [{ data: deptEntries.map(e => e[1]), backgroundColor: COLORS }]
        },
        options: { responsive: true, maintainAspectRatio: false, cutout: '55%', plugins: { legend: { position: 'right', labels: { color: '#94a3b8' } } } }
    });
}

// ========== TIEMPOS ==========
async function cargarTiempos() {
    const desde = document.getElementById('fechaDesde').value;
    const hasta = document.getElementById('fechaHasta').value;
    const d = await api('/reportes/tiempos-atencion?desde=' + desde + '&hasta=' + hasta);
    
    if (!d) return;
    
    document.getElementById('kpiPrimeraResp').textContent = d.primera_respuesta_promedio ? d.primera_respuesta_promedio : '—';
    document.getElementById('kpiResolucion').textContent = d.tiempo_resolucion_promedio ? d.tiempo_resolucion_promedio : '—';
    document.getElementById('kpiMenos24').textContent = d.tickets_menos_24h || 0;
    document.getElementById('kpiMas48').textContent = d.tickets_mas_48h || 0;
    document.getElementById('kpiMenos24Pct').textContent = d.total_resueltos > 0 ? Math.round((d.tickets_menos_24h / d.total_resueltos) * 100) + '%' : '0%';
    document.getElementById('kpiMas48Pct').textContent = d.total_resueltos > 0 ? Math.round((d.tickets_mas_48h / d.total_resueltos) * 100) + '%' : '0%';
    
    const tbody = document.getElementById('tiemposTabla');
    if (d.detalle && d.detalle.length) {
        const prioColors = { critica:'#ef4444', alta:'#f97316', media:'#eab308', baja:'#22c55e' };
        tbody.innerHTML = d.detalle.map(t => `
            <tr style="cursor:pointer" onclick="window.location='ticket-detalle.html?id=${t.id}'">
                <td><code style="color:var(--color-primary);font-family:var(--font-mono);font-size:12px">${t.numero_ticket}</code></td>
                <td>${t.titulo}</td>
                <td><span style="color:${prioColors[t.prioridad]||'#94a3b8'}">${t.prioridad}</span></td>
                <td>${t.tecnico_nombre || '—'}</td>
                <td><strong>${t.tiempo_total_horas != null ? t.tiempo_total_horas + 'h' : '—'}</strong></td>
                <td>${t.primera_respuesta_horas != null ? t.primera_respuesta_horas + 'h' : '—'}</td>
            </tr>
        `).join('');
    } else {
        tbody.innerHTML = '<tr><td colspan="6" style="text-align:center;padding:40px;color:var(--text-muted)">No hay datos</td></tr>';
    }
}

// ========== SLA ==========
async function cargarSLA() {
    const d = await api('/reportes/sla');
    
    // KPIs
    let totalDentro = 0, totalFuera = 0, totalVencidos = 0;
    d.forEach(r => { totalDentro += parseInt(r.dentro_sla || 0); totalFuera += parseInt(r.fuera_sla || 0); totalVencidos += parseInt(r.vencidos_abiertos || 0); });
    const pctGlobal = (totalDentro + totalFuera) > 0 ? Math.round((totalDentro / (totalDentro + totalFuera)) * 100) : 100;
    document.getElementById('slaKpis').innerHTML = `
        <div class="kpi-card"><div class="kpi-label">Cumplimiento Global</div><div class="kpi-value ${pctGlobal>=90?'green':pctGlobal>=70?'yellow':'red'}">${pctGlobal}%</div></div>
        <div class="kpi-card"><div class="kpi-label">Dentro de SLA</div><div class="kpi-value green">${totalDentro}</div></div>
        <div class="kpi-card"><div class="kpi-label">Fuera de SLA</div><div class="kpi-value red">${totalFuera}</div></div>
        <div class="kpi-card"><div class="kpi-label">Vencidos Abiertos</div><div class="kpi-value ${totalVencidos>0?'red':'green'}">${totalVencidos}</div></div>
    `;

    // Chart
    destroyChart('sla');
    charts.sla = new Chart(document.getElementById('chartSLA'), {
        type: 'bar',
        data: {
            labels: d.map(r => r.categoria || 'Sin cat.'),
            datasets: [
                { label: 'Dentro SLA', data: d.map(r => r.dentro_sla), backgroundColor: '#22c55e' },
                { label: 'Fuera SLA', data: d.map(r => r.fuera_sla), backgroundColor: '#ef4444' },
                { label: 'Vencidos abiertos', data: d.map(r => r.vencidos_abiertos), backgroundColor: '#eab308' }
            ]
        },
        options: { ...CHART_DEFAULTS, plugins: { ...CHART_DEFAULTS.plugins }, scales: { ...CHART_DEFAULTS.scales, x: { ...CHART_DEFAULTS.scales.x, stacked: true }, y: { ...CHART_DEFAULTS.scales.y, stacked: true } } }
    });

    // Table
    let html = '<table class="tech-table"><tr><th>Categoría</th><th>SLA (h)</th><th>Total</th><th>Cumplimiento</th></tr>';
    d.forEach(r => {
        const pct = r.total > 0 ? Math.round((r.dentro_sla / r.total) * 100) : 100;
        const cls = pct >= 90 ? 'sla-ok' : pct >= 70 ? 'sla-warn' : 'sla-bad';
        html += `<tr><td>${r.categoria || '—'}</td><td>${r.sla_horas || 24}h</td><td>${r.total}</td><td><span class="sla-badge ${cls}">${pct}%</span></td></tr>`;
    });
    html += '</table>';
    document.getElementById('slaTabla').innerHTML = html;
}

// ========== TÉCNICOS ==========
async function cargarTecnicos() {
    const d = await api('/reportes/rendimiento-tecnicos');

    // Table
    let html = '<table class="tech-table"><tr><th>Técnico</th><th>Asignados</th><th>Resueltos</th><th>Pendientes</th><th>Prom. (h)</th><th>Satisf.</th></tr>';
    d.forEach(r => {
        const sat = r.satisfaccion ? parseFloat(r.satisfaccion).toFixed(1) : '—';
        const prom = r.promedio_horas ? parseFloat(r.promedio_horas).toFixed(1) : '—';
        html += `<tr><td><strong>${r.nombre_completo}</strong></td><td>${r.total_asignados}</td><td style="color:#22c55e">${r.resueltos}</td><td style="color:#eab308">${r.pendientes}</td><td>${prom}</td><td>${sat !== '—' ? '<span class="star">★</span> '+sat : '—'}</td></tr>`;
    });
    html += '</table>';
    document.getElementById('techTabla').innerHTML = html;

    // Bar chart
    destroyChart('tecnicos');
    const tecData = d.filter(r => r.total_asignados > 0);
    charts.tecnicos = new Chart(document.getElementById('chartTecnicos'), {
        type: 'bar',
        data: {
            labels: tecData.map(r => r.nombre_completo.split(' ')[0]),
            datasets: [
                { label: 'Resueltos', data: tecData.map(r => r.resueltos), backgroundColor: '#22c55e' },
                { label: 'Pendientes', data: tecData.map(r => r.pendientes), backgroundColor: '#eab308' }
            ]
        },
        options: { ...CHART_DEFAULTS, indexAxis: 'y' }
    });

    // Satisfaction chart
    destroyChart('satisfaccion');
    const satData = d.filter(r => r.satisfaccion);
    if (satData.length > 0) {
        charts.satisfaccion = new Chart(document.getElementById('chartSatisfaccion'), {
            type: 'bar',
            data: {
                labels: satData.map(r => r.nombre_completo.split(' ')[0]),
                datasets: [{ label: 'Satisfacción', data: satData.map(r => parseFloat(r.satisfaccion)), backgroundColor: '#eab308', borderRadius: 6 }]
            },
            options: { ...CHART_DEFAULTS, scales: { ...CHART_DEFAULTS.scales, y: { ...CHART_DEFAULTS.scales.y, max: 5 } } }
        });
    }
}

// ========== ACTIVOS ==========
async function cargarActivos() {
    const d = await api('/reportes/activos-resumen');

    document.getElementById('activoKpis').innerHTML = `
        <div class="kpi-card"><div class="kpi-label">Total Activos</div><div class="kpi-value blue">${d.total_activos || 0}</div></div>
        <div class="kpi-card"><div class="kpi-label">Activos Operativos</div><div class="kpi-value green">${d.activos_activos || 0}</div></div>
        <div class="kpi-card"><div class="kpi-label">Valor Total Inventario</div><div class="kpi-value">$${Number(d.valor_total||0).toLocaleString()}</div></div>
        <div class="kpi-card"><div class="kpi-label">Garantías Vencidas</div><div class="kpi-value ${d.garantias_vencidas>0?'red':'green'}">${d.garantias_vencidas || 0}</div><div class="kpi-sub">${d.garantias_por_vencer || 0} por vencer (90d)</div></div>
    `;

    const coloresTipo = { desktop:'#3b82f6', laptop:'#22c55e', servidor:'#ef4444', impresora:'#8b5cf6', red:'#f97316', monitor:'#06b6d4', periferico:'#ec4899', otro:'#64748b' };
    const coloresEstado = { activo:'#22c55e', mantenimiento:'#eab308', baja:'#ef4444', bodega:'#64748b' };

    destroyChart('activosTipo');
    if (d.por_tipo && d.por_tipo.length) {
        charts.activosTipo = new Chart(document.getElementById('chartActivosTipo'), {
            type: 'doughnut',
            data: { labels: d.por_tipo.map(r => r.tipo), datasets: [{ data: d.por_tipo.map(r => r.total), backgroundColor: d.por_tipo.map(r => coloresTipo[r.tipo] || '#64748b') }] },
            options: { responsive: true, maintainAspectRatio: false, cutout: '55%', plugins: { legend: { position: 'right', labels: { color: '#94a3b8' } } } }
        });
    }

    destroyChart('activosEstado');
    if (d.por_estado && d.por_estado.length) {
        charts.activosEstado = new Chart(document.getElementById('chartActivosEstado'), {
            type: 'doughnut',
            data: { labels: d.por_estado.map(r => r.estado), datasets: [{ data: d.por_estado.map(r => r.total), backgroundColor: d.por_estado.map(r => coloresEstado[r.estado] || '#64748b') }] },
            options: { responsive: true, maintainAspectRatio: false, cutout: '55%', plugins: { legend: { position: 'right', labels: { color: '#94a3b8' } } } }
        });
    }

    destroyChart('activosDepto');
    if (d.por_departamento && d.por_departamento.length) {
        charts.activosDepto = new Chart(document.getElementById('chartActivosDepto'), {
            type: 'bar',
            data: {
                labels: d.por_departamento.map(r => r.departamento),
                datasets: [{ label: 'Activos', data: d.por_departamento.map(r => r.total), backgroundColor: COLORS, borderRadius: 6 }]
            },
            options: { ...CHART_DEFAULTS, plugins: { ...CHART_DEFAULTS.plugins, legend: { display: false } } }
        });
    }
}

// ========== EXPORT ==========
function exportarCSV() {
    const desde = document.getElementById('fechaDesde').value;
    const hasta = document.getElementById('fechaHasta').value;
    const token = NexusAuth.getToken();
    window.open('/api/v1/reportes/exportar-tickets?desde=' + desde + '&hasta=' + hasta + '&token=' + token, '_blank');
}
</script>
</body>
</html>