<!DOCTYPE html>
<html lang="es">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Nexus IT — Panel Supervisor</title>
<link href="https://fonts.googleapis.com/css2?family=DM+Sans:wght@300;400;500;600;700;800&family=JetBrains+Mono:wght@400;600&display=swap" rel="stylesheet">
<link rel="stylesheet" href="../assets/css/app.css">
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<style>
.kpi-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(170px,1fr)); gap: 12px; margin-bottom: 20px; }
.kpi { background: var(--bg-card); border: 1px solid var(--border-color); border-radius: 12px; padding: 18px; }
.kpi .value { font-size: 28px; font-weight: 800; line-height: 1; }
.kpi .label { font-size: 11px; color: var(--text-muted); margin-top: 6px; text-transform: uppercase; letter-spacing: 0.3px; }
.kpi .sub { font-size: 11px; color: var(--text-muted); margin-top: 3px; }
.kpi.warn .value { color: #f59e0b; }
.kpi.danger .value { color: #ef4444; }
.kpi.success .value { color: #22c55e; }
.kpi.info .value { color: #6366f1; }
.kpi.blue .value { color: #3b82f6; }

.grid-2 { display: grid; grid-template-columns: 1fr 1fr; gap: 16px; margin-bottom: 16px; }
.grid-3 { display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 16px; margin-bottom: 16px; }
@media(max-width:900px){ .grid-2,.grid-3{grid-template-columns:1fr;} }

.card { background: var(--bg-card); border: 1px solid var(--border-color); border-radius: 12px; padding: 20px; margin-bottom: 16px; }
.card-title { font-size: 15px; font-weight: 700; margin-bottom: 14px; color: var(--text-primary); }
.card-subtitle { font-size: 12px; color: var(--text-muted); margin-top: -10px; margin-bottom: 14px; }

.tech-table { width: 100%; border-collapse: collapse; }
.tech-table th { font-size: 10px; text-transform: uppercase; letter-spacing: 0.5px; color: var(--text-muted); text-align: left; padding: 8px 6px; border-bottom: 1px solid var(--border-color); }
.tech-table td { padding: 10px 6px; font-size: 13px; border-bottom: 1px solid var(--border-color); }
.tech-table tr:last-child td { border-bottom: none; }
.tech-table tr:hover { background: rgba(255,255,255,0.02); }

.tec-row { display: flex; align-items: center; gap: 12px; padding: 10px 0; border-bottom: 1px solid var(--border-color); }
.tec-row:last-child { border-bottom: none; }
.tec-name { font-weight: 600; flex: 1; font-size: 13px; }
.tec-stat { text-align: center; min-width: 55px; }
.tec-stat .n { font-size: 16px; font-weight: 700; }
.tec-stat .l { font-size: 9px; color: var(--text-muted); text-transform: uppercase; }
.stars { color: #f59e0b; }

.tab-group { display: flex; gap: 3px; background: rgba(255,255,255,0.04); border-radius: 8px; padding: 3px; margin-bottom: 16px; flex-wrap: wrap; }
.tab-btn { padding: 6px 14px; border-radius: 6px; font-size: 12px; cursor: pointer; border: none; background: transparent; color: var(--text-muted); font-family: inherit; font-weight: 500; }
.tab-btn.active { background: var(--primary); color: #fff; }
.tab-btn:hover:not(.active) { background: rgba(255,255,255,0.06); }

.chart-card { background: var(--bg-card); border: 1px solid var(--border-color); border-radius: 12px; padding: 18px; }
.chart-card h3 { font-size: 13px; font-weight: 600; color: var(--text-primary); margin-bottom: 12px; }
.chart-card canvas { max-height: 260px; }

.badge { display: inline-block; padding: 2px 8px; border-radius: 6px; font-size: 11px; font-weight: 600; }
.badge-open { background: rgba(59,130,246,0.15); color: #3b82f6; }
.badge-progress { background: rgba(234,179,8,0.15); color: #eab308; }
.badge-pending { background: rgba(139,92,246,0.15); color: #8b5cf6; }
.badge-resolved { background: rgba(34,197,94,0.15); color: #22c55e; }
.badge-closed { background: rgba(100,116,139,0.15); color: #64748b; }

.prio-dot { display: inline-block; width: 8px; height: 8px; border-radius: 50%; margin-right: 4px; }
.prio-critica { background: #ef4444; }
.prio-alta { background: #f97316; }
.prio-media { background: #eab308; }
.prio-baja { background: #22c55e; }

.link-row { cursor: pointer; transition: background 0.15s; }
.link-row:hover { background: rgba(99,102,241,0.06) !important; }

.empty-state { text-align: center; padding: 40px; color: var(--text-muted); font-size: 13px; }
</style>
</head>
<body>
<div class="app-layout">
  <aside class="sidebar" id="sidebar"></aside>
  <main class="main-content">
    <header class="top-bar">
      <div class="top-bar-left">
        <button class="btn-menu-mobile" onclick="document.getElementById('sidebar').classList.toggle('open')">
          <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><line x1="4" y1="12" x2="20" y2="12"/><line x1="4" y1="6" x2="20" y2="6"/><line x1="4" y1="18" x2="20" y2="18"/></svg>
        </button>
        <h1 class="page-title">Panel Supervisor</h1>
      </div>
      <div class="top-bar-right">
        <button class="btn btn-sm" onclick="cargarTodo()" title="Actualizar"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="23 4 23 10 17 10"/><path d="M20.49 15a9 9 0 1 1-2.12-9.36L23 10"/></svg> Actualizar</button>
      </div>
    </header>

    <div class="content-area">

      <!-- KPIs principales -->
      <div class="kpi-grid" id="kpis">
        <div class="kpi"><div class="value">—</div><div class="label">Cargando...</div></div>
      </div>

      <!-- TABS -->
      <div class="tab-group">
        <button class="tab-btn active" data-tab="overview" onclick="switchTab('overview')">Vista General</button>
        <button class="tab-btn" data-tab="tickets" onclick="switchTab('tickets')">Todos los Tickets</button>
        <button class="tab-btn" data-tab="tecnicos" onclick="switchTab('tecnicos')">Carga Técnicos</button>
        <button class="tab-btn" data-tab="solicitantes" onclick="switchTab('solicitantes')">Top Solicitantes</button>
        <button class="tab-btn" data-tab="departamentos" onclick="switchTab('departamentos')">Departamentos</button>
      </div>

      <!-- TAB: OVERVIEW -->
      <div id="tab-overview" class="tab-content">
        <div class="grid-2">
          <div class="chart-card">
            <h3>Tendencia SLA (12 meses)</h3>
            <canvas id="chartSLA"></canvas>
          </div>
          <div class="chart-card">
            <h3>CSAT Mensual</h3>
            <canvas id="chartCSAT"></canvas>
          </div>
        </div>
        <div class="grid-2">
          <div class="card">
            <h3 class="card-title">Tickets Críticos / Vencidos</h3>
            <div id="ticketsCriticos"><div class="empty-state">Cargando...</div></div>
          </div>
          <div class="card">
            <h3 class="card-title">Reaperturas Recientes</h3>
            <div id="reaperturas"><div class="empty-state">Cargando...</div></div>
          </div>
        </div>
      </div>

      <!-- TAB: TODOS LOS TICKETS -->
      <div id="tab-tickets" class="tab-content" style="display:none">
        <div class="card">
          <div style="display:flex;gap:8px;margin-bottom:14px;flex-wrap:wrap;align-items:center">
            <input type="text" id="buscarTicket" placeholder="Buscar ticket..." style="background:var(--bg-card);border:1px solid var(--border-color);color:var(--text-primary);padding:6px 10px;border-radius:8px;font-size:13px;width:200px;">
            <select id="filtroEstado" style="background:var(--bg-card);border:1px solid var(--border-color);color:var(--text-primary);padding:6px 8px;border-radius:8px;font-size:13px;">
              <option value="">Todos los estados</option>
              <option value="abierto">Abierto</option>
              <option value="en_progreso">En Progreso</option>
              <option value="pendiente">Pendiente</option>
              <option value="resuelto">Resuelto</option>
              <option value="cerrado">Cerrado</option>
            </select>
            <select id="filtroPrioridad" style="background:var(--bg-card);border:1px solid var(--border-color);color:var(--text-primary);padding:6px 8px;border-radius:8px;font-size:13px;">
              <option value="">Todas las prioridades</option>
              <option value="critica">Crítica</option>
              <option value="alta">Alta</option>
              <option value="media">Media</option>
              <option value="baja">Baja</option>
            </select>
            <button class="btn btn-primary btn-sm" onclick="cargarTickets()">Filtrar</button>
          </div>
          <table class="tech-table">
            <thead>
              <tr>
                <th>Ticket</th>
                <th>Título</th>
                <th>Solicitante</th>
                <th>Técnico</th>
                <th>Prioridad</th>
                <th>Estado</th>
                <th>Creado</th>
              </tr>
            </thead>
            <tbody id="ticketsTabla"><tr><td colspan="7" class="empty-state">Cargando...</td></tr></tbody>
          </table>
          <div id="ticketsPaginacion" style="display:flex;justify-content:center;gap:8px;margin-top:14px"></div>
        </div>
      </div>

      <!-- TAB: CARGA TÉCNICOS -->
      <div id="tab-tecnicos" class="tab-content" style="display:none">
        <div class="card">
          <h3 class="card-title">Carga de Trabajo por Técnico</h3>
          <p class="card-subtitle">Tickets abiertos, resueltos en el mes, vencidos y satisfacción</p>
          <div id="tecnicosLista"><div class="empty-state">Cargando...</div></div>
        </div>
        <div class="grid-2" style="margin-top:16px">
          <div class="chart-card">
            <h3>Tickets por Técnico</h3>
            <canvas id="chartTecnicos"></canvas>
          </div>
          <div class="chart-card">
            <h3>Satisfacción por Técnico</h3>
            <canvas id="chartSatTecnicos"></canvas>
          </div>
        </div>
      </div>

      <!-- TAB: TOP SOLICITANTES -->
      <div id="tab-solicitantes" class="tab-content" style="display:none">
        <div class="card">
          <h3 class="card-title">Top Solicitantes (últimos 90 días)</h3>
          <p class="card-subtitle">Usuarios que más tickets han creado</p>
          <table class="tech-table">
            <thead><tr><th>Usuario</th><th>Departamento</th><th>Total</th><th>Urgentes</th><th>Reabiertos</th></tr></thead>
            <tbody id="solicitantesTabla"><tr><td colspan="5" class="empty-state">Cargando...</td></tr></tbody>
          </table>
        </div>
      </div>

      <!-- TAB: DEPARTAMENTOS -->
      <div id="tab-departamentos" class="tab-content" style="display:none">
        <div class="card">
          <h3 class="card-title">Usuarios por Departamento</h3>
          <p class="card-subtitle">Actividad, equipos asignados y último acceso</p>
          <div id="departamentosLista"><div class="empty-state">Cargando...</div></div>
        </div>
      </div>

    </div>
  </main>
</div>

<script src="../assets/js/api.js"></script>
<script src="../assets/js/auth.js"></script>
<script src="../assets/js/sidebar.js"></script>
<script src="../assets/js/notifications.js"></script>
<script src="../assets/js/utils.js"></script>
<script>
if (!NexusAuth.requireAuth(['admin','supervisor'])) throw 'No auth';

const charts = {};
const COLORS = ['#6366f1','#3b82f6','#22c55e','#eab308','#ef4444','#f97316','#8b5cf6','#06b6d4','#ec4899','#64748b'];
let ticketPage = 1;

function destroyChart(id) { if (charts[id]) { charts[id].destroy(); delete charts[id]; } }

function switchTab(tab) {
    document.querySelectorAll('.tab-content').forEach(t => t.style.display = 'none');
    document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
    document.getElementById('tab-' + tab).style.display = 'block';
    document.querySelector(`[data-tab="${tab}"]`).classList.add('active');
    if (tab === 'tickets' && !document.getElementById('ticketsTabla').dataset.loaded) cargarTickets();
    if (tab === 'tecnicos') cargarTecnicos();
    if (tab === 'solicitantes') cargarSolicitantes();
    if (tab === 'departamentos') cargarDepartamentos();
}

function badgeEstado(e) {
    const m = { abierto:'badge-open', en_progreso:'badge-progress', pendiente:'badge-pending', resuelto:'badge-resolved', cerrado:'badge-closed' };
    return `<span class="badge ${m[e]||''}">${(e||'').replace('_',' ')}</span>`;
}

// ========== KPIs ==========
async function cargarKPIs() {
    try {
        const res = await NexusAPI.get('/supervisor/resumen');
        if (!res.success) return;
        const d = res.data;
        document.getElementById('kpis').innerHTML = `
            <div class="kpi blue"><div class="value">${d.tickets_abiertos}</div><div class="label">Tickets Abiertos</div></div>
            <div class="kpi danger"><div class="value">${d.tickets_vencidos}</div><div class="label">Vencidos SLA</div></div>
            <div class="kpi warn"><div class="value">${d.por_vencer_2h}</div><div class="label">Por Vencer (2h)</div></div>
            <div class="kpi ${d.sla_cumplimiento>=90?'success':d.sla_cumplimiento>=70?'warn':'danger'}"><div class="value">${d.sla_cumplimiento}%</div><div class="label">Cumplimiento SLA</div></div>
            <div class="kpi info"><div class="value">${d.csat_promedio||'—'}</div><div class="label">CSAT Promedio</div><div class="sub">${d.csat_total} calificaciones</div></div>
            <div class="kpi"><div class="value">${d.resolucion_promedio_h}h</div><div class="label">Resolución Prom.</div></div>
            <div class="kpi"><div class="value">${d.tickets_mes}</div><div class="label">Tickets este Mes</div><div class="sub">${d.tickets_hoy} hoy / ${d.tickets_semana} semana</div></div>
            <div class="kpi ${d.tickets_reabiertos>0?'warn':''}"><div class="value">${d.tickets_reabiertos}</div><div class="label">Reabiertos (30d)</div></div>
        `;
    } catch(e) { console.error(e); }
}

// ========== OVERVIEW CHARTS ==========
async function cargarOverview() {
    // SLA Tendencia
    try {
        const res = await NexusAPI.get('/supervisor/sla-tendencia');
        if (res.success && res.data && res.data.length) {
            destroyChart('sla');
            charts.sla = new Chart(document.getElementById('chartSLA'), {
                type: 'line',
                data: {
                    labels: res.data.map(r => r.mes),
                    datasets: [{
                        label: '% Cumplimiento',
                        data: res.data.map(r => r.total > 0 ? Math.round((r.dentro_sla / r.total) * 100) : 100),
                        borderColor: '#22c55e', backgroundColor: 'rgba(34,197,94,0.1)',
                        fill: true, tension: 0.4
                    }]
                },
                options: { responsive: true, maintainAspectRatio: false, scales: { y: { min: 0, max: 100, grid: { color: 'rgba(100,116,139,0.1)' } }, x: { grid: { display: false } } }, plugins: { legend: { display: false } } }
            });
            // CSAT
            destroyChart('csat');
            const csatData = res.data.filter(r => r.csat);
            if (csatData.length) {
                charts.csat = new Chart(document.getElementById('chartCSAT'), {
                    type: 'bar',
                    data: {
                        labels: csatData.map(r => r.mes),
                        datasets: [{ label: 'CSAT', data: csatData.map(r => parseFloat(r.csat)), backgroundColor: '#eab308', borderRadius: 6 }]
                    },
                    options: { responsive: true, maintainAspectRatio: false, scales: { y: { min: 0, max: 5, grid: { color: 'rgba(100,116,139,0.1)' } }, x: { grid: { display: false } } }, plugins: { legend: { display: false } } }
                });
            }
        }
    } catch(e) {}

    // Tickets críticos / vencidos
    try {
        const res = await NexusAPI.get('/tickets?estado=abierto&prioridad=critica&per_page=5');
        const res2 = await NexusAPI.get('/tickets?estado=en_progreso&prioridad=critica&per_page=5');
        const tickets = [...(res.data?.data||[]), ...(res2.data?.data||[])].slice(0, 8);
        if (tickets.length) {
            document.getElementById('ticketsCriticos').innerHTML = tickets.map(t => `
                <div class="tec-row link-row" onclick="window.location='ticket-detalle.html?id=${t.id}'">
                    <div style="flex:1">
                        <code style="color:var(--primary);font-size:11px">${t.numero_ticket}</code>
                        <div style="font-size:13px;font-weight:500;margin-top:2px">${t.titulo}</div>
                        <div style="font-size:11px;color:var(--text-muted)">${t.solicitante_nombre||''} → ${t.tecnico_nombre||'Sin asignar'}</div>
                    </div>
                    <div style="text-align:right">
                        <span class="prio-dot prio-${t.prioridad}"></span>${t.prioridad}
                        <div style="font-size:10px;color:var(--text-muted);margin-top:2px">${NexusUtils.timeAgo(t.created_at)}</div>
                    </div>
                </div>
            `).join('');
        } else {
            document.getElementById('ticketsCriticos').innerHTML = '<div class="empty-state">Sin tickets críticos abiertos</div>';
        }
    } catch(e) {}

    // Reaperturas
    try {
        const res = await NexusAPI.get('/tickets?per_page=10');
        const reabiertos = (res.data?.data||[]).filter(t => t.reaberturas > 0).slice(0, 6);
        if (reabiertos.length) {
            document.getElementById('reaperturas').innerHTML = reabiertos.map(t => `
                <div class="tec-row link-row" onclick="window.location='ticket-detalle.html?id=${t.id}'">
                    <div style="flex:1">
                        <code style="color:#f59e0b;font-size:11px">${t.numero_ticket}</code>
                        <div style="font-size:13px;margin-top:2px">${t.titulo}</div>
                    </div>
                    <div style="text-align:right;font-size:12px">
                        <strong style="color:#f59e0b">${t.reaberturas}x</strong> reabierto
                    </div>
                </div>
            `).join('');
        } else {
            document.getElementById('reaperturas').innerHTML = '<div class="empty-state">Sin reaperturas recientes</div>';
        }
    } catch(e) {}
}

// ========== TODOS LOS TICKETS ==========
async function cargarTickets(page) {
    ticketPage = page || 1;
    const buscar = document.getElementById('buscarTicket').value;
    const estado = document.getElementById('filtroEstado').value;
    const prioridad = document.getElementById('filtroPrioridad').value;
    let url = `/tickets?page=${ticketPage}&per_page=20`;
    if (buscar) url += `&buscar=${encodeURIComponent(buscar)}`;
    if (estado) url += `&estado=${estado}`;
    if (prioridad) url += `&prioridad=${prioridad}`;

    try {
        const res = await NexusAPI.get(url);
        if (!res.success) return;
        const tbody = document.getElementById('ticketsTabla');
        tbody.dataset.loaded = '1';
        const tickets = res.data?.data || res.data || [];

        if (!tickets.length) {
            tbody.innerHTML = '<tr><td colspan="7" class="empty-state">No hay tickets</td></tr>';
            return;
        }

        tbody.innerHTML = tickets.map(t => `
            <tr class="link-row" onclick="window.location='ticket-detalle.html?id=${t.id}'">
                <td><code style="color:var(--primary);font-family:var(--font-mono);font-size:11px">${t.numero_ticket}</code></td>
                <td style="max-width:200px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap">${t.titulo}</td>
                <td>${t.solicitante_nombre||'—'}</td>
                <td>${t.tecnico_nombre||'<span style="color:#ef4444">Sin asignar</span>'}</td>
                <td><span class="prio-dot prio-${t.prioridad}"></span>${t.prioridad}</td>
                <td>${badgeEstado(t.estado)}</td>
                <td style="font-size:11px;color:var(--text-muted)">${NexusUtils.timeAgo(t.created_at)}</td>
            </tr>
        `).join('');

        // Paginación
        const meta = res.data?.pagination || {};
        const totalPages = meta.total_pages || 1;
        let pagHtml = '';
        if (totalPages > 1) {
            if (ticketPage > 1) pagHtml += `<button class="btn btn-sm" onclick="cargarTickets(${ticketPage-1})">← Ant</button>`;
            pagHtml += `<span style="color:var(--text-muted);font-size:12px;padding:6px">Pág ${ticketPage} de ${totalPages}</span>`;
            if (ticketPage < totalPages) pagHtml += `<button class="btn btn-sm" onclick="cargarTickets(${ticketPage+1})">Sig →</button>`;
        }
        document.getElementById('ticketsPaginacion').innerHTML = pagHtml;
    } catch(e) { console.error(e); }
}

// ========== TÉCNICOS ==========
async function cargarTecnicos() {
    try {
        const res = await NexusAPI.get('/supervisor/carga-tecnicos');
        if (!res.success) return;
        const d = res.data || [];

        document.getElementById('tecnicosLista').innerHTML = d.length ? d.map(t => `
            <div class="tec-row">
                <div class="tec-name">${t.nombre_completo}<br><span style="font-size:10px;color:var(--text-muted)">${t.email}</span></div>
                <div class="tec-stat"><div class="n" style="color:#3b82f6">${t.tickets_abiertos||0}</div><div class="l">Abiertos</div></div>
                <div class="tec-stat"><div class="n" style="color:#22c55e">${t.resueltos_mes||0}</div><div class="l">Resueltos</div></div>
                <div class="tec-stat"><div class="n" style="color:${(t.vencidos||0)>0?'#ef4444':'inherit'}">${t.vencidos||0}</div><div class="l">Vencidos</div></div>
                <div class="tec-stat"><div class="n">${t.promedio_resolucion_h ? Math.round(t.promedio_resolucion_h)+'h' : '—'}</div><div class="l">Prom.</div></div>
                <div class="tec-stat"><div class="n stars">${t.csat ? parseFloat(t.csat).toFixed(1) : '—'}</div><div class="l">CSAT</div></div>
            </div>
        `).join('') : '<div class="empty-state">Sin datos de técnicos</div>';

        // Charts
        const tecData = d.filter(t => (t.tickets_abiertos||0) + (t.resueltos_mes||0) > 0);
        if (tecData.length) {
            destroyChart('tecnicos');
            charts.tecnicos = new Chart(document.getElementById('chartTecnicos'), {
                type: 'bar',
                data: {
                    labels: tecData.map(t => t.nombre_completo.split(' ')[0]),
                    datasets: [
                        { label: 'Abiertos', data: tecData.map(t => t.tickets_abiertos||0), backgroundColor: '#3b82f6' },
                        { label: 'Resueltos', data: tecData.map(t => t.resueltos_mes||0), backgroundColor: '#22c55e' },
                        { label: 'Vencidos', data: tecData.map(t => t.vencidos||0), backgroundColor: '#ef4444' }
                    ]
                },
                options: { responsive: true, maintainAspectRatio: false, indexAxis: 'y', scales: { x: { grid: { color: 'rgba(100,116,139,0.1)' }, stacked: true }, y: { grid: { display: false }, stacked: true } }, plugins: { legend: { labels: { color: '#94a3b8', font: { size: 11 } } } } }
            });
            destroyChart('satTec');
            const satData = tecData.filter(t => t.csat);
            if (satData.length) {
                charts.satTec = new Chart(document.getElementById('chartSatTecnicos'), {
                    type: 'bar',
                    data: {
                        labels: satData.map(t => t.nombre_completo.split(' ')[0]),
                        datasets: [{ label: 'CSAT', data: satData.map(t => parseFloat(t.csat)), backgroundColor: '#eab308', borderRadius: 6 }]
                    },
                    options: { responsive: true, maintainAspectRatio: false, scales: { y: { min: 0, max: 5, grid: { color: 'rgba(100,116,139,0.1)' } }, x: { grid: { display: false } } }, plugins: { legend: { display: false } } }
                });
            }
        }
    } catch(e) { console.error(e); }
}

// ========== TOP SOLICITANTES ==========
async function cargarSolicitantes() {
    try {
        const res = await NexusAPI.get('/supervisor/top-solicitantes');
        if (!res.success) return;
        const d = res.data || [];
        const tbody = document.getElementById('solicitantesTabla');
        tbody.innerHTML = d.length ? d.map(s => `
            <tr>
                <td><strong>${s.nombre_completo}</strong><br><span style="font-size:11px;color:var(--text-muted)">${s.email}</span></td>
                <td>${s.departamento||'—'}</td>
                <td><strong>${s.total_tickets}</strong></td>
                <td style="color:${(s.tickets_urgentes||0)>0?'#ef4444':'inherit'}">${s.tickets_urgentes||0}</td>
                <td style="color:${(s.tickets_reabiertos||0)>0?'#f59e0b':'inherit'}">${s.tickets_reabiertos||0}</td>
            </tr>
        `).join('') : '<tr><td colspan="5" class="empty-state">Sin datos</td></tr>';
    } catch(e) { console.error(e); }
}

// ========== DEPARTAMENTOS ==========
async function cargarDepartamentos() {
    try {
        const res = await NexusAPI.get('/supervisor/usuarios-departamento');
        if (!res.success) return;
        const d = res.data || [];
        const agrupado = {};
        d.forEach(u => {
            const depto = u.departamento || 'Sin departamento';
            if (!agrupado[depto]) agrupado[depto] = [];
            agrupado[depto].push(u);
        });

        let html = '';
        Object.keys(agrupado).sort().forEach(depto => {
            const users = agrupado[depto];
            const totalTickets = users.reduce((s, u) => s + (parseInt(u.tickets_mes)||0), 0);
            html += `<div style="margin-bottom:20px">`;
            html += `<h4 style="margin:0 0 10px;color:var(--primary);font-size:14px;display:flex;justify-content:space-between">${depto} <span style="font-size:12px;color:var(--text-muted)">${users.length} usuarios · ${totalTickets} tickets/mes</span></h4>`;
            html += '<table class="tech-table"><tr><th>Usuario</th><th>Email</th><th>Tickets Mes</th><th>Equipos</th><th>Último Acceso</th></tr>';
            users.forEach(u => {
                html += `<tr>
                    <td><strong>${u.nombre_completo}</strong></td>
                    <td style="color:var(--text-muted);font-size:12px">${u.email}</td>
                    <td>${u.tickets_mes||0}</td>
                    <td>${u.equipos_asignados||0}</td>
                    <td style="font-size:11px;color:var(--text-muted)">${u.ultimo_login ? NexusUtils.timeAgo(u.ultimo_login) : 'Nunca'}</td>
                </tr>`;
            });
            html += '</table></div>';
        });
        document.getElementById('departamentosLista').innerHTML = html || '<div class="empty-state">Sin datos</div>';
    } catch(e) { console.error(e); }
}

// ========== INIT ==========
async function cargarTodo() {
    await cargarKPIs();
    await cargarOverview();
}

// Enter para buscar tickets
document.getElementById('buscarTicket')?.addEventListener('keyup', e => { if (e.key === 'Enter') cargarTickets(); });

cargarTodo();
</script>
</body>
</html>