<?php
/**
 * NEXUS IT - Alerta Controller v3
 * Centro de Operaciones - Muestra estado operativo completo
 * Incluye: situación actual + alertas que requieren atención
 */

class AlertaController {
    
    /**
     * GET /alertas/centro - Centro operativo unificado
     */
    public static function centro(): void {
        $auth = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($auth);
        $db = getDB();
        
        $alertas = [];
        
        // Helper: detectar columna
        $colCache = [];
        $colExists = function(string $table, string $col) use ($db, &$colCache) {
            $key = "$table.$col";
            if (!isset($colCache[$key])) {
                try { $s = $db->query("SHOW COLUMNS FROM `$table` LIKE '$col'"); $colCache[$key] = $s->rowCount() > 0; }
                catch(Exception $e) { $colCache[$key] = false; }
            }
            return $colCache[$key];
        };
        
        // Helper: tabla existe
        $tableExists = function(string $table) use ($db) {
            try { $db->query("SELECT 1 FROM `$table` LIMIT 1"); return true; }
            catch(Exception $e) { return false; }
        };
        
        // ══════════════════════════════════════════════
        // 1. TICKETS - SLA VENCIDO
        // ══════════════════════════════════════════════
        if ($colExists('tickets', 'fecha_vencimiento')) {
            try {
                $stmt = $db->query("
                    SELECT t.id, t.numero_ticket, t.titulo, t.prioridad, t.fecha_vencimiento,
                           u.nombre_completo AS tecnico,
                           TIMESTAMPDIFF(HOUR, t.fecha_vencimiento, NOW()) AS horas_vencido
                    FROM tickets t LEFT JOIN usuarios u ON t.tecnico_asignado_id = u.id
                    WHERE t.estado NOT IN ('cerrado','resuelto')
                    AND t.fecha_vencimiento IS NOT NULL AND t.fecha_vencimiento < NOW()
                    ORDER BY t.fecha_vencimiento ASC LIMIT 50
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $h = max(1, (int)$r['horas_vencido']);
                    $alertas[] = [
                        'id'=>'sla_'.$r['id'],'tipo'=>'sla_vencido','categoria'=>'tickets',
                        'severidad'=>$h>24?'critical':'warning',
                        'titulo'=>"SLA vencido: {$r['numero_ticket']}",
                        'mensaje'=>"{$r['titulo']} — vencido hace {$h}h".($r['tecnico']?" ({$r['tecnico']})":' (sin asignar)'),
                        'referencia'=>['tipo'=>'ticket','id'=>$r['id']],
                        'fecha'=>$r['fecha_vencimiento'],
                        'accion'=>'Ver ticket','accion_url'=>"ticket-detalle.html?id={$r['id']}"
                    ];
                }
            } catch(Exception $e) {}
            
            // SLA próximo a vencer (< 2h)
            try {
                $stmt = $db->query("
                    SELECT t.id, t.numero_ticket, t.titulo, t.prioridad, t.fecha_vencimiento,
                           u.nombre_completo AS tecnico,
                           TIMESTAMPDIFF(MINUTE, NOW(), t.fecha_vencimiento) AS min_rest
                    FROM tickets t LEFT JOIN usuarios u ON t.tecnico_asignado_id = u.id
                    WHERE t.estado NOT IN ('cerrado','resuelto')
                    AND t.fecha_vencimiento IS NOT NULL AND t.fecha_vencimiento > NOW()
                    AND t.fecha_vencimiento < DATE_ADD(NOW(), INTERVAL 2 HOUR)
                    ORDER BY t.fecha_vencimiento ASC LIMIT 20
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $m = max(1,(int)$r['min_rest']);
                    $alertas[] = [
                        'id'=>'sla_prox_'.$r['id'],'tipo'=>'sla_proximo','categoria'=>'tickets',
                        'severidad'=>$m<30?'critical':'warning',
                        'titulo'=>"SLA por vencer: {$r['numero_ticket']}",
                        'mensaje'=>"{$r['titulo']} — vence en {$m} min".($r['tecnico']?" ({$r['tecnico']})":' (sin asignar)'),
                        'referencia'=>['tipo'=>'ticket','id'=>$r['id']],
                        'fecha'=>$r['fecha_vencimiento'],
                        'accion'=>'Ver ticket','accion_url'=>"ticket-detalle.html?id={$r['id']}"
                    ];
                }
            } catch(Exception $e) {}
        }
        
        // 2. TICKETS SIN ASIGNAR (> 1 hora)
        try {
            $stmt = $db->query("
                SELECT t.id, t.numero_ticket, t.titulo, t.prioridad, t.created_at,
                       TIMESTAMPDIFF(HOUR, t.created_at, NOW()) AS horas
                FROM tickets t
                WHERE t.estado = 'abierto' AND t.tecnico_asignado_id IS NULL
                AND t.created_at < DATE_SUB(NOW(), INTERVAL 1 HOUR)
                ORDER BY t.created_at ASC LIMIT 20
            ");
            foreach ($stmt->fetchAll() as $r) {
                $h = max(1,(int)$r['horas']);
                $alertas[] = [
                    'id'=>'sin_asignar_'.$r['id'],'tipo'=>'ticket_sin_asignar','categoria'=>'tickets',
                    'severidad'=>$h>8?'critical':'warning',
                    'titulo'=>"Sin asignar: {$r['numero_ticket']}",
                    'mensaje'=>"{$r['titulo']} — {$h}h sin técnico",
                    'referencia'=>['tipo'=>'ticket','id'=>$r['id']],
                    'fecha'=>$r['created_at'],
                    'accion'=>'Asignar','accion_url'=>"ticket-detalle.html?id={$r['id']}"
                ];
            }
        } catch(Exception $e) {}
        
        // ══════════════════════════════════════════════
        // 3. MANTENIMIENTOS VENCIDOS (fecha pasó y no se completaron)
        // ══════════════════════════════════════════════
        if ($tableExists('mantenimientos')) {
            try {
                $stmt = $db->query("
                    SELECT m.id, m.titulo, m.tipo, m.estado, m.fecha_programada,
                           a.codigo_activo, a.hostname, u.nombre_completo AS tecnico,
                           DATEDIFF(CURDATE(), m.fecha_programada) AS dias_vencido
                    FROM mantenimientos m
                    LEFT JOIN activos a ON m.activo_id = a.id
                    LEFT JOIN usuarios u ON m.tecnico_id = u.id
                    WHERE m.estado IN ('programado','en_curso') AND m.fecha_programada < CURDATE()
                    ORDER BY m.fecha_programada ASC LIMIT 30
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $d = max(1,(int)$r['dias_vencido']);
                    $eq = ($r['codigo_activo']??'Sin equipo').($r['hostname']?" ({$r['hostname']})":'');
                    $alertas[] = [
                        'id'=>'mant_vencido_'.$r['id'],'tipo'=>'mantenimiento_vencido','categoria'=>'mantenimientos',
                        'severidad'=>$d>7?'critical':'warning',
                        'titulo'=>"Mant. vencido: {$r['titulo']}",
                        'mensaje'=>ucfirst($r['tipo'])." en {$eq} — vencido hace {$d} día(s)".($r['tecnico']?" ({$r['tecnico']})":''),
                        'referencia'=>['tipo'=>'mantenimiento','id'=>$r['id']],
                        'fecha'=>$r['fecha_programada'],
                        'accion'=>'Ver','accion_url'=>"mantenimientos.html"
                    ];
                }
            } catch(Exception $e) {}
            
            // 4. MANTENIMIENTOS HOY
            try {
                $stmt = $db->query("
                    SELECT m.id, m.titulo, m.tipo, m.fecha_programada,
                           a.codigo_activo, a.hostname, u.nombre_completo AS tecnico
                    FROM mantenimientos m
                    LEFT JOIN activos a ON m.activo_id = a.id
                    LEFT JOIN usuarios u ON m.tecnico_id = u.id
                    WHERE m.estado = 'programado' AND m.fecha_programada = CURDATE()
                    ORDER BY m.fecha_programada ASC LIMIT 20
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $eq = ($r['codigo_activo']??'Sin equipo').($r['hostname']?" ({$r['hostname']})":'');
                    $alertas[] = [
                        'id'=>'mant_hoy_'.$r['id'],'tipo'=>'mantenimiento_hoy','categoria'=>'mantenimientos',
                        'severidad'=>'warning',
                        'titulo'=>"Mant. HOY: {$r['titulo']}",
                        'mensaje'=>ucfirst($r['tipo'])." en {$eq}".($r['tecnico']?" — Técnico: {$r['tecnico']}":''),
                        'referencia'=>['tipo'=>'mantenimiento','id'=>$r['id']],
                        'fecha'=>$r['fecha_programada'],
                        'accion'=>'Ver','accion_url'=>"mantenimientos.html"
                    ];
                }
            } catch(Exception $e) {}
            
            // 5. MANTENIMIENTOS PRÓXIMOS (1-30 días) → info
            try {
                $stmt = $db->query("
                    SELECT m.id, m.titulo, m.tipo, m.fecha_programada, m.recurrencia,
                           a.codigo_activo, a.hostname, u.nombre_completo AS tecnico,
                           DATEDIFF(m.fecha_programada, CURDATE()) AS dias
                    FROM mantenimientos m
                    LEFT JOIN activos a ON m.activo_id = a.id
                    LEFT JOIN usuarios u ON m.tecnico_id = u.id
                    WHERE m.estado = 'programado'
                    AND m.fecha_programada > CURDATE()
                    AND m.fecha_programada <= DATE_ADD(CURDATE(), INTERVAL 30 DAY)
                    ORDER BY m.fecha_programada ASC LIMIT 20
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $d = (int)$r['dias'];
                    $eq = ($r['codigo_activo']??'Sin equipo').($r['hostname']?" ({$r['hostname']})":'');
                    $sev = $d <= 3 ? 'warning' : 'info';
                    $alertas[] = [
                        'id'=>'mant_prox_'.$r['id'],'tipo'=>'mantenimiento_proximo','categoria'=>'mantenimientos',
                        'severidad'=>$sev,
                        'titulo'=>"Mant. en {$d}d: {$r['titulo']}",
                        'mensaje'=>ucfirst($r['tipo'])." en {$eq} — ".date('d/m/Y',strtotime($r['fecha_programada'])).($r['tecnico']?" ({$r['tecnico']})":'').($r['recurrencia']&&$r['recurrencia']!=='unica'?" [{$r['recurrencia']}]":''),
                        'referencia'=>['tipo'=>'mantenimiento','id'=>$r['id']],
                        'fecha'=>$r['fecha_programada'],
                        'accion'=>'Ver','accion_url'=>"mantenimientos.html"
                    ];
                }
            } catch(Exception $e) {}
            
            // 6. MANTENIMIENTOS EN CURSO
            try {
                $stmt = $db->query("
                    SELECT m.id, m.titulo, m.tipo, m.fecha_programada,
                           a.codigo_activo, a.hostname, u.nombre_completo AS tecnico
                    FROM mantenimientos m
                    LEFT JOIN activos a ON m.activo_id = a.id
                    LEFT JOIN usuarios u ON m.tecnico_id = u.id
                    WHERE m.estado = 'en_curso'
                    ORDER BY m.fecha_programada ASC LIMIT 20
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $eq = ($r['codigo_activo']??'Sin equipo').($r['hostname']?" ({$r['hostname']})":'');
                    $alertas[] = [
                        'id'=>'mant_curso_'.$r['id'],'tipo'=>'mantenimiento_en_curso','categoria'=>'mantenimientos',
                        'severidad'=>'info',
                        'titulo'=>"En curso: {$r['titulo']}",
                        'mensaje'=>ucfirst($r['tipo'])." en {$eq}".($r['tecnico']?" — {$r['tecnico']}":''),
                        'referencia'=>['tipo'=>'mantenimiento','id'=>$r['id']],
                        'fecha'=>$r['fecha_programada'],
                        'accion'=>'Ver','accion_url'=>"mantenimientos.html"
                    ];
                }
            } catch(Exception $e) {}
        }
        
        // ══════════════════════════════════════════════
        // 7. AGENTE - Equipos online/offline y alertas
        // ══════════════════════════════════════════════
        
        // Equipos con agente activo (última conexión < 20 min) → info positiva
        $hasAgentToken = $colExists('activos', 'agent_token');
        $lastConnCol = $colExists('activos', 'ultima_conexion') ? 'ultima_conexion' : ($colExists('activos', 'ultimo_reporte') ? 'ultimo_reporte' : null);
        
        if ($hasAgentToken && $lastConnCol) {
            try {
                $stmt = $db->query("
                    SELECT id, codigo_activo, hostname, marca, modelo, $lastConnCol AS last_conn,
                           TIMESTAMPDIFF(MINUTE, $lastConnCol, NOW()) AS mins_ago
                    FROM activos
                    WHERE agent_token IS NOT NULL AND $lastConnCol IS NOT NULL
                    ORDER BY $lastConnCol DESC LIMIT 30
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $mins = (int)$r['mins_ago'];
                    $eq = $r['codigo_activo'].($r['hostname']?" ({$r['hostname']})":'');
                    if ($mins > 60) {
                        // Offline > 1 hora
                        $h = round($mins/60);
                        $alertas[] = [
                            'id'=>'offline_'.$r['id'],'tipo'=>'equipo_offline','categoria'=>'agente',
                            'severidad'=>$h>24?'critical':'warning',
                            'titulo'=>"Offline: {$eq}",
                            'mensaje'=>($r['marca']??'').' '.($r['modelo']??'')." — sin reportar hace ".($h>24?round($h/24).'d':$h.'h'),
                            'referencia'=>['tipo'=>'activo','id'=>$r['id']],
                            'fecha'=>$r['last_conn'],
                            'accion'=>'Ver activo','accion_url'=>"activo-detalle.html?id={$r['id']}"
                        ];
                    } else {
                        // Online → info
                        $alertas[] = [
                            'id'=>'online_'.$r['id'],'tipo'=>'equipo_online','categoria'=>'agente',
                            'severidad'=>'ok',
                            'titulo'=>"Online: {$eq}",
                            'mensaje'=>($r['marca']??'').' '.($r['modelo']??'')." — reportó hace {$mins} min",
                            'referencia'=>['tipo'=>'activo','id'=>$r['id']],
                            'fecha'=>$r['last_conn'],
                            'accion'=>'Ver activo','accion_url'=>"activo-detalle.html?id={$r['id']}"
                        ];
                    }
                }
            } catch(Exception $e) {}
        }
        
        // Alertas tabla agente_alertas
        if ($tableExists('agente_alertas')) {
            try {
                $stmt = $db->query("
                    SELECT aa.id, aa.tipo, aa.mensaje, aa.severidad, aa.created_at,
                           a.codigo_activo, a.hostname
                    FROM agente_alertas aa LEFT JOIN activos a ON aa.activo_id = a.id
                    WHERE aa.resuelta = 0 ORDER BY aa.created_at DESC LIMIT 30
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $eq = ($r['codigo_activo']??'').($r['hostname']?" ({$r['hostname']})":'');
                    $alertas[] = [
                        'id'=>'agente_'.$r['id'],'tipo'=>'agente_'.$r['tipo'],'categoria'=>'agente',
                        'severidad'=>$r['severidad'],
                        'titulo'=>ucfirst(str_replace('_',' ',$r['tipo'])).": {$eq}",
                        'mensaje'=>$r['mensaje'],
                        'referencia'=>['tipo'=>'agente_alerta','id'=>$r['id']],
                        'fecha'=>$r['created_at'],
                        'accion'=>'Resolver','accion_url'=>null,'resolvable'=>true
                    ];
                }
            } catch(Exception $e) {}
        }
        
        // Alertas tabla legacy
        if ($tableExists('alertas')) {
            try {
                $stmt = $db->query("
                    SELECT al.id, al.tipo_alerta, al.mensaje, al.severidad, al.created_at,
                           a.codigo_activo, a.hostname
                    FROM alertas al LEFT JOIN activos a ON al.activo_id = a.id
                    WHERE al.resuelta = 0 ORDER BY al.created_at DESC LIMIT 30
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $eq = ($r['codigo_activo']??'').($r['hostname']?" ({$r['hostname']})":'');
                    $alertas[] = [
                        'id'=>'legacy_'.$r['id'],'tipo'=>$r['tipo_alerta'],'categoria'=>'agente',
                        'severidad'=>$r['severidad'],
                        'titulo'=>ucfirst(str_replace('_',' ',$r['tipo_alerta'])).": {$eq}",
                        'mensaje'=>$r['mensaje'],
                        'referencia'=>['tipo'=>'alerta','id'=>$r['id']],
                        'fecha'=>$r['created_at'],
                        'accion'=>'Resolver','accion_url'=>null,'resolvable'=>true
                    ];
                }
            } catch(Exception $e) {}
        }
        
        // ══════════════════════════════════════════════
        // 8. GARANTÍAS POR VENCER (30 días)
        // ══════════════════════════════════════════════
        if ($colExists('activos', 'garantia_hasta')) {
            try {
                $stmt = $db->query("
                    SELECT id, codigo_activo, hostname, marca, modelo, garantia_hasta,
                           DATEDIFF(garantia_hasta, CURDATE()) AS dias
                    FROM activos WHERE estado = 'activo' AND garantia_hasta IS NOT NULL
                    AND garantia_hasta BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)
                    ORDER BY garantia_hasta ASC
                ");
                foreach ($stmt->fetchAll() as $r) {
                    $d = (int)$r['dias'];
                    $alertas[] = [
                        'id'=>'garantia_'.$r['id'],'tipo'=>'garantia_vence','categoria'=>'activos',
                        'severidad'=>$d<=7?'warning':'info',
                        'titulo'=>"Garantía: {$r['codigo_activo']}",
                        'mensaje'=>($r['marca']??'').' '.($r['modelo']??'')." — vence ".date('d/m/Y',strtotime($r['garantia_hasta']))." ({$d}d)",
                        'referencia'=>['tipo'=>'activo','id'=>$r['id']],
                        'fecha'=>$r['garantia_hasta'],
                        'accion'=>'Ver activo','accion_url'=>"activo-detalle.html?id={$r['id']}"
                    ];
                }
            } catch(Exception $e) {}
        }
        
        // ══════════════════════════════════════════════
        // ORDENAR: critical > warning > info > ok
        // ══════════════════════════════════════════════
        $sevOrder = ['critical'=>0,'warning'=>1,'info'=>2,'ok'=>3];
        usort($alertas, function($a,$b) use ($sevOrder) {
            $sa=$sevOrder[$a['severidad']]??4; $sb=$sevOrder[$b['severidad']]??4;
            if($sa!==$sb) return $sa-$sb;
            return strtotime($b['fecha']??'2000-01-01') - strtotime($a['fecha']??'2000-01-01');
        });
        
        // RESUMEN
        $resumen = [
            'total'=>count($alertas),
            'critical'=>count(array_filter($alertas,fn($a)=>$a['severidad']==='critical')),
            'warning'=>count(array_filter($alertas,fn($a)=>$a['severidad']==='warning')),
            'info'=>count(array_filter($alertas,fn($a)=>$a['severidad']==='info')),
            'ok'=>count(array_filter($alertas,fn($a)=>$a['severidad']==='ok')),
            'tickets'=>count(array_filter($alertas,fn($a)=>$a['categoria']==='tickets')),
            'mantenimientos'=>count(array_filter($alertas,fn($a)=>$a['categoria']==='mantenimientos')),
            'agente'=>count(array_filter($alertas,fn($a)=>$a['categoria']==='agente')),
            'activos'=>count(array_filter($alertas,fn($a)=>$a['categoria']==='activos')),
        ];
        
        // Filtros
        $fc = $_GET['categoria'] ?? null;
        $fs = $_GET['severidad'] ?? null;
        if ($fc) $alertas = array_values(array_filter($alertas, fn($a) => $a['categoria'] === $fc));
        if ($fs) $alertas = array_values(array_filter($alertas, fn($a) => $a['severidad'] === $fs));
        
        Response::success(['resumen'=>$resumen,'alertas'=>$alertas]);
    }
    
    /** GET /alertas */
    public static function index(): void {
        $auth = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($auth);
        $db = getDB();
        $page=max(1,intval($_GET['page']??1)); $pp=min(MAX_PAGE_SIZE,max(1,intval($_GET['per_page']??DEFAULT_PAGE_SIZE))); $off=($page-1)*$pp;
        $w=[]; $p=[];
        if(isset($_GET['resuelta'])){$w[]='al.resuelta=?';$p[]=intval($_GET['resuelta']);}
        if(!empty($_GET['severidad'])){$w[]='al.severidad=?';$p[]=$_GET['severidad'];}
        if(!empty($_GET['tipo_alerta'])){$w[]='al.tipo_alerta=?';$p[]=$_GET['tipo_alerta'];}
        $ws=$w?'WHERE '.implode(' AND ',$w):'';
        $s=$db->prepare("SELECT COUNT(*) FROM alertas al $ws");$s->execute($p);$t=(int)$s->fetchColumn();
        $s=$db->prepare("SELECT al.*,a.codigo_activo,a.hostname AS activo_hostname,r.nombre_completo AS resuelta_por_nombre FROM alertas al LEFT JOIN activos a ON al.activo_id=a.id LEFT JOIN usuarios r ON al.resuelta_por_id=r.id $ws ORDER BY al.resuelta ASC,al.severidad DESC,al.created_at DESC LIMIT $pp OFFSET $off");
        $s->execute($p);Response::paginated($s->fetchAll(),$t,$page,$pp);
    }
    
    /** PUT /alertas/{id}/resolver */
    public static function resolver(int $id): void {
        $auth = AuthMiddleware::verificar(); RoleMiddleware::requireTecnico($auth); $db = getDB(); $ok=false;
        try{$s=$db->prepare("SELECT id FROM alertas WHERE id=? AND resuelta=0");$s->execute([$id]);
            if($s->fetch()){$db->prepare("UPDATE alertas SET resuelta=1,resuelta_por_id=?,resuelta_at=NOW() WHERE id=?")->execute([$auth['id'],$id]);$ok=true;}}catch(Exception $e){}
        if(!$ok){try{$s=$db->prepare("SELECT id FROM agente_alertas WHERE id=? AND resuelta=0");$s->execute([$id]);
            if($s->fetch()){$db->prepare("UPDATE agente_alertas SET resuelta=1 WHERE id=?")->execute([$id]);$ok=true;}}catch(Exception $e){}}
        if(!$ok) Response::notFound('Alerta no encontrada');
        Response::success(null,'Alerta resuelta');
    }
    
    /** PUT /alertas/resolver-masivo */
    public static function resolverMasivo(): void {
        $auth = AuthMiddleware::verificar(); RoleMiddleware::requireTecnico($auth);
        $data=json_decode(file_get_contents('php://input'),true)??[]; $db=getDB();
        if(empty($data['ids'])||!is_array($data['ids'])) Response::error('Se requiere lista de IDs');
        $c=0;
        foreach($data['ids'] as $raw){
            if(is_string($raw)&&str_starts_with($raw,'agente_')){$id=intval(substr($raw,7));try{$db->prepare("UPDATE agente_alertas SET resuelta=1 WHERE id=? AND resuelta=0")->execute([$id]);$c++;}catch(Exception $e){}}
            elseif(is_string($raw)&&str_starts_with($raw,'legacy_')){$id=intval(substr($raw,7));try{$db->prepare("UPDATE alertas SET resuelta=1,resuelta_por_id=?,resuelta_at=NOW() WHERE id=? AND resuelta=0")->execute([$auth['id'],$id]);$c++;}catch(Exception $e){}}
            else{$id=intval($raw);try{$db->prepare("UPDATE alertas SET resuelta=1,resuelta_por_id=?,resuelta_at=NOW() WHERE id=?")->execute([$auth['id'],$id]);$c++;}catch(Exception $e){}}
        }
        Response::success(['resueltas'=>$c],'Alertas resueltas');
    }
    
    /** GET /alertas/resumen */
    public static function resumen(): void {
        $auth = AuthMiddleware::verificar(); RoleMiddleware::requireTecnico($auth); $db = getDB();
        try{$s=$db->query("SELECT tipo_alerta,severidad,COUNT(*) AS cantidad FROM alertas WHERE resuelta=0 GROUP BY tipo_alerta,severidad");Response::success($s->fetchAll());}
        catch(Exception $e){Response::success([]);}
    }
}
