<?php
class ContratoController {

    // GET /contratos
    public static function index(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $stmt = $db->query("SELECT c.*,
            (SELECT COUNT(*) FROM contrato_activos ca WHERE ca.contrato_id = c.id) as total_activos,
            CASE WHEN c.fecha_fin < CURDATE() THEN 'vencido'
                 WHEN c.fecha_fin <= DATE_ADD(CURDATE(), INTERVAL c.alerta_dias DAY) THEN 'por_vencer'
                 ELSE 'vigente' END as estado_contrato
            FROM contratos_soporte c WHERE c.activo = 1 ORDER BY c.fecha_fin ASC");
        Response::success($stmt->fetchAll());
    }

    // GET /contratos/:id
    public static function ver(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $stmt = $db->prepare("SELECT * FROM contratos_soporte WHERE id = ?");
        $stmt->execute([$id]);
        $contrato = $stmt->fetch();
        if (!$contrato) Response::notFound('Contrato no encontrado');

        $activos = $db->prepare("SELECT a.id, a.nombre, a.codigo_interno, a.tipo, a.marca, a.modelo
            FROM contrato_activos ca JOIN activos a ON ca.activo_id = a.id WHERE ca.contrato_id = ?");
        $activos->execute([$id]);
        $contrato['activos'] = $activos->fetchAll();
        Response::success($contrato);
    }

    // POST /contratos
    public static function crear(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireSupervisor($payload);
        $db = getDB();
        $body = Validator::getBody();
        if (empty($body['nombre']) || empty($body['proveedor'])) Response::error('Nombre y proveedor requeridos');

        $stmt = $db->prepare("INSERT INTO contratos_soporte (nombre, proveedor, tipo, numero_contrato, fecha_inicio, fecha_fin, costo, moneda, cobertura, contacto_nombre, contacto_email, contacto_telefono, alerta_dias, notas) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
        $stmt->execute([
            $body['nombre'], $body['proveedor'], $body['tipo'] ?? 'garantia',
            $body['numero_contrato'] ?? null, $body['fecha_inicio'], $body['fecha_fin'],
            $body['costo'] ?? null, $body['moneda'] ?? 'COP',
            $body['cobertura'] ?? null, $body['contacto_nombre'] ?? null,
            $body['contacto_email'] ?? null, $body['contacto_telefono'] ?? null,
            $body['alerta_dias'] ?? 30, $body['notas'] ?? null
        ]);
        Response::success(['id' => (int)$db->lastInsertId()], 'Contrato creado');
    }

    // PUT /contratos/:id
    public static function actualizar(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireSupervisor($payload);
        $db = getDB();
        $body = Validator::getBody();
        $fields = []; $params = [];
        foreach (['nombre','proveedor','tipo','numero_contrato','fecha_inicio','fecha_fin','costo','moneda','cobertura','contacto_nombre','contacto_email','contacto_telefono','alerta_dias','notas','activo'] as $f) {
            if (array_key_exists($f, $body)) { $fields[] = "$f = ?"; $params[] = $body[$f]; }
        }
        if (empty($fields)) Response::error('Nada que actualizar');
        $params[] = $id;
        $db->prepare("UPDATE contratos_soporte SET " . implode(', ', $fields) . " WHERE id = ?")->execute($params);
        Response::success(null, 'Contrato actualizado');
    }

    // POST /contratos/:id/activos — vincular activo
    public static function vincularActivo(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $body = Validator::getBody();
        if (empty($body['activo_id'])) Response::error('activo_id requerido');

        try {
            $db->prepare("INSERT INTO contrato_activos (contrato_id, activo_id) VALUES (?,?)")
                ->execute([$id, $body['activo_id']]);
            Response::success(null, 'Activo vinculado al contrato');
        } catch (\Exception $e) {
            Response::error('El activo ya está vinculado a este contrato');
        }
    }

    // DELETE /contratos/:id/activos
    public static function desvincularActivo(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $body = Validator::getBody();
        $db->prepare("DELETE FROM contrato_activos WHERE contrato_id = ? AND activo_id = ?")
            ->execute([$id, $body['activo_id'] ?? 0]);
        Response::success(null, 'Activo desvinculado');
    }
}
