<?php
class LicenciaController {

    // GET /licencias
    public static function index(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $stmt = $db->query("SELECT l.*, 
            (SELECT COUNT(*) FROM licencia_asignaciones la WHERE la.licencia_id = l.id) as asignadas,
            CASE WHEN l.fecha_vencimiento IS NOT NULL AND l.fecha_vencimiento < CURDATE() THEN 'vencida'
                 WHEN l.fecha_vencimiento IS NOT NULL AND l.fecha_vencimiento <= DATE_ADD(CURDATE(), INTERVAL l.alerta_renovacion_dias DAY) THEN 'por_vencer'
                 ELSE 'vigente' END as estado_licencia
            FROM licencias_software l WHERE l.activo = 1 ORDER BY l.nombre");
        Response::success($stmt->fetchAll());
    }

    // GET /licencias/:id
    public static function ver(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $stmt = $db->prepare("SELECT l.* FROM licencias_software l WHERE l.id = ?");
        $stmt->execute([$id]);
        $lic = $stmt->fetch();
        if (!$lic) Response::notFound('Licencia no encontrada');

        // Asignaciones
        $asig = $db->prepare("SELECT la.*, a.hostname as activo_nombre, a.codigo_activo as codigo_interno,
            u.nombre_completo as usuario_nombre
            FROM licencia_asignaciones la
            LEFT JOIN activos a ON la.activo_id = a.id
            LEFT JOIN usuarios u ON la.usuario_id = u.id
            WHERE la.licencia_id = ? ORDER BY la.created_at DESC");
        $asig->execute([$id]);
        $lic['asignaciones'] = $asig->fetchAll();
        Response::success($lic);
    }

    // POST /licencias
    public static function crear(): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireSupervisor($payload);
        $db = getDB();
        $body = Validator::getBody();
        if (empty($body['nombre'])) Response::error('Nombre requerido');

        $stmt = $db->prepare("INSERT INTO licencias_software (nombre, editor, tipo, clave_licencia, cantidad_total, costo_unitario, fecha_compra, fecha_vencimiento, proveedor, notas, alerta_renovacion_dias) VALUES (?,?,?,?,?,?,?,?,?,?,?)");
        $stmt->execute([
            $body['nombre'], $body['editor'] ?? null, $body['tipo'] ?? 'suscripcion',
            $body['clave_licencia'] ?? null, $body['cantidad_total'] ?? 1,
            $body['costo_unitario'] ?? null, $body['fecha_compra'] ?? null,
            $body['fecha_vencimiento'] ?? null, $body['proveedor'] ?? null,
            $body['notas'] ?? null, $body['alerta_renovacion_dias'] ?? 30
        ]);
        Response::success(['id' => (int)$db->lastInsertId()], 'Licencia creada');
    }

    // PUT /licencias/:id
    public static function actualizar(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireSupervisor($payload);
        $db = getDB();
        $body = Validator::getBody();
        $fields = []; $params = [];
        foreach (['nombre','editor','tipo','clave_licencia','cantidad_total','costo_unitario','fecha_compra','fecha_vencimiento','proveedor','notas','alerta_renovacion_dias','activo'] as $f) {
            if (array_key_exists($f, $body)) { $fields[] = "$f = ?"; $params[] = $body[$f]; }
        }
        if (empty($fields)) Response::error('Nada que actualizar');
        $params[] = $id;
        $db->prepare("UPDATE licencias_software SET " . implode(', ', $fields) . " WHERE id = ?")->execute($params);

        // Recalcular usadas
        $count = (int)$db->prepare("SELECT COUNT(*) FROM licencia_asignaciones WHERE licencia_id = ?")->execute([$id]);
        $db->prepare("UPDATE licencias_software SET cantidad_usada = (SELECT COUNT(*) FROM licencia_asignaciones WHERE licencia_id = ?) WHERE id = ?")->execute([$id, $id]);

        Response::success(null, 'Licencia actualizada');
    }

    // POST /licencias/:id/asignar
    public static function asignar(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $body = Validator::getBody();

        // Verificar disponibilidad
        $lic = $db->prepare("SELECT cantidad_total, cantidad_usada FROM licencias_software WHERE id = ?")->execute([$id]);
        $lic = $db->prepare("SELECT cantidad_total, cantidad_usada FROM licencias_software WHERE id = ?");
        $lic->execute([$id]);
        $licData = $lic->fetch();
        if (!$licData) Response::notFound('Licencia no encontrada');
        if ($licData['cantidad_usada'] >= $licData['cantidad_total']) Response::error('No hay licencias disponibles');

        $db->prepare("INSERT INTO licencia_asignaciones (licencia_id, activo_id, usuario_id, notas) VALUES (?,?,?,?)")
            ->execute([$id, $body['activo_id'] ?? null, $body['usuario_id'] ?? null, $body['notas'] ?? null]);

        $db->prepare("UPDATE licencias_software SET cantidad_usada = cantidad_usada + 1 WHERE id = ?")->execute([$id]);
        Response::success(null, 'Licencia asignada');
    }

    // DELETE /licencias/:id/asignacion/:asigId
    public static function desasignar(int $id): void {
        $payload = AuthMiddleware::verificar();
        RoleMiddleware::requireTecnico($payload);
        $db = getDB();
        $body = Validator::getBody();
        $asigId = $body['asignacion_id'] ?? 0;

        $db->prepare("DELETE FROM licencia_asignaciones WHERE id = ? AND licencia_id = ?")->execute([$asigId, $id]);
        $db->prepare("UPDATE licencias_software SET cantidad_usada = GREATEST(0, cantidad_usada - 1) WHERE id = ?")->execute([$id]);
        Response::success(null, 'Asignacin removida');
    }
}