<?php
/**
 * Configuración centralizada de la aplicación
 * Gestiona acceso a variables de configuración
 */
class AppConfig {
    private static $config = null;
    
    /**
     * Cargar configuración desde base de datos
     */
    private static function load(): void {
        if (self::$config !== null) return;
        
        try {
            $db = getDB();
            $stmt = $db->query("SELECT clave, valor FROM configuracion WHERE grupo = 'general'");
            self::$config = [];
            foreach ($stmt->fetchAll() as $row) {
                self::$config[$row['clave']] = $row['valor'];
            }
        } catch (\Exception $e) {
            error_log("AppConfig: Error cargando configuración - " . $e->getMessage());
            self::$config = [];
        }
    }
    
    /**
     * Obtener valor de configuración
     * Prioridad: BD > ENV > Default
     */
    public static function get(string $key, $default = null) {
        self::load();
        
        // Primero intentar desde BD
        if (isset(self::$config[$key])) {
            return self::$config[$key];
        }
        
        // Luego desde variables de entorno
        if (isset($_ENV[$key])) {
            return $_ENV[$key];
        }
        
        // Finalmente desde getenv
        $envValue = getenv($key);
        if ($envValue !== false) {
            return $envValue;
        }
        
        // Default
        return $default;
    }
    
    /**
     * Obtener URL base de la aplicación
     */
    public static function getBaseUrl(): string {
        return rtrim(self::get('base_url', self::get('BASE_URL', 'https://ana.evolucionamos.com')), '/');
    }
    
    /**
     * Construir URL completa de la app
     */
    public static function getAppUrl(string $path): string {
        return self::getBaseUrl() . '/' . ltrim($path, '/');
    }
    
    /**
     * Construir URL completa de la API
     */
    public static function getApiUrl(string $path): string {
        return self::getBaseUrl() . '/api/v1/' . ltrim($path, '/');
    }
    
    /**
     * Verificar si está en modo debug
     */
    public static function isDebug(): bool {
        $debug = self::get('APP_DEBUG', 'false');
        return $debug === 'true' || $debug === '1' || $debug === true;
    }
}