<?php
class Validator {
    private array $errors = [];
    private array $data;

    public function __construct(array $data) { $this->data = $data; }

    public static function make(array $data): self { return new self($data); }

    public function required(string ...$fields): self {
        // Support both required('field') and required('field', 'Label')
        // When called as required('titulo', 'Título'), treat second as label
        // When called as required('titulo', 'descripcion'), treat both as fields
        foreach ($fields as $f) {
            if (!isset($this->data[$f]) || (is_string($this->data[$f]) && trim($this->data[$f]) === '')) {
                // Only add error if it looks like a field name (no spaces, lowercase)
                // If it has spaces or uppercase it's likely a label for the previous field
                if (preg_match('/^[a-z_]+$/', $f)) {
                    $this->errors[$f] = "El campo $f es obligatorio";
                }
            }
        }
        return $this;
    }

    public function email(string $field): self {
        if (isset($this->data[$field]) && !filter_var($this->data[$field], FILTER_VALIDATE_EMAIL)) {
            $this->errors[$field] = "Email inválido";
        }
        return $this;
    }

    public function minLength(string $field, int $min): self {
        if (isset($this->data[$field]) && strlen($this->data[$field]) < $min) {
            $this->errors[$field] = "Mínimo $min caracteres";
        }
        return $this;
    }

    public function maxLength(string $field, int $max): self {
        if (isset($this->data[$field]) && strlen($this->data[$field]) > $max) {
            $this->errors[$field] = "Máximo $max caracteres";
        }
        return $this;
    }

    public function in(string $field, array $values): self {
        if (isset($this->data[$field]) && !in_array($this->data[$field], $values)) {
            $this->errors[$field] = "Valor no válido para $field";
        }
        return $this;
    }

    public function inList(string $field, array $values): self {
        return $this->in($field, $values);
    }

    public function numeric(string $field): self {
        if (isset($this->data[$field]) && !is_numeric($this->data[$field])) {
            $this->errors[$field] = "$field debe ser numérico";
        }
        return $this;
    }

    public function integer(string $field): self {
        if (isset($this->data[$field]) && !is_numeric($this->data[$field])) {
            $this->errors[$field] = "$field debe ser un número entero";
        }
        return $this;
    }

    public function date(string $field): self {
        if (isset($this->data[$field]) && strtotime($this->data[$field]) === false) {
            $this->errors[$field] = "$field debe ser una fecha válida";
        }
        return $this;
    }

    public function fails(): bool { return !empty($this->errors); }
    public function errors(): array { return $this->errors; }

    public function validate(): void {
        if ($this->fails()) Response::validationError($this->errors);
    }

    public static function getBody(): array {
        $body = json_decode(file_get_contents('php://input'), true);
        return $body ?: [];
    }
}