<?php
/**
 * NEXUS IT - Cron: Limpieza y Depreciación
 * Ejecutar mensualmente via cPanel Cron Jobs
 * Comando: php /home/USUARIO/public_html/ana/cron/cleanup_and_depreciation.php
 */

require_once dirname(__DIR__) . '/api/config/app.php';
require_once dirname(__DIR__) . '/api/config/database.php';

$db = getDB();

echo "[" . date('Y-m-d H:i:s') . "] Limpieza y depreciación...\n";

// ========== 1. LIMPIAR TELEMETRÍA ANTIGUA (>90 días) ==========
$stmt = $db->exec("DELETE FROM telemetria WHERE created_at < DATE_SUB(NOW(), INTERVAL 90 DAY)");
echo "Telemetría eliminada: registros antiguos limpiados\n";

// Limpiar log de actividad antiguo (>180 días)
$db->exec("DELETE FROM log_actividad WHERE created_at < DATE_SUB(NOW(), INTERVAL 180 DAY)");

// Limpiar rate limiter
$cacheDir = '/tmp/nexus_rate_limit';
if (is_dir($cacheDir)) {
    foreach (glob("$cacheDir/*") as $file) {
        if (filemtime($file) < time() - 3600) unlink($file);
    }
}

// ========== 2. CALCULAR DEPRECIACIÓN ==========
echo "Calculando depreciación...\n";

// Obtener activos sin registro de depreciación y crearlos
$db->exec("
    INSERT IGNORE INTO depreciacion (activo_id, metodo, vida_util_meses, valor_residual, depreciacion_mensual, valor_actual)
    SELECT 
        a.id,
        'linea_recta',
        36,
        0,
        CASE WHEN a.costo_adquisicion > 0 THEN ROUND(a.costo_adquisicion / 36, 2) ELSE 0 END,
        a.costo_adquisicion
    FROM activos a
    LEFT JOIN depreciacion d ON a.id = d.activo_id
    WHERE d.id IS NULL AND a.costo_adquisicion > 0 AND a.fecha_compra IS NOT NULL
");

// Actualizar depreciación mensual
$stmt = $db->query("
    SELECT d.*, a.costo_adquisicion, a.fecha_compra
    FROM depreciacion d
    INNER JOIN activos a ON d.activo_id = a.id
    WHERE a.estado != 'baja' AND a.costo_adquisicion > 0 AND a.fecha_compra IS NOT NULL
");

$actualizados = 0;
$stmtUpdate = $db->prepare("
    UPDATE depreciacion SET valor_actual = ?, depreciacion_mensual = ?, ultimo_calculo = CURDATE() WHERE id = ?
");

while ($row = $stmt->fetch()) {
    $mesesTranscurridos = max(0, (int)((time() - strtotime($row['fecha_compra'])) / (30 * 86400)));
    $costoOriginal = (float)$row['costo_adquisicion'];
    $vidaUtil = (int)$row['vida_util_meses'];
    $valorResidual = (float)$row['valor_residual'];
    
    if ($vidaUtil <= 0) continue;
    
    if ($row['metodo'] === 'linea_recta') {
        $depMensual = ($costoOriginal - $valorResidual) / $vidaUtil;
        $depAcumulada = min($mesesTranscurridos * $depMensual, $costoOriginal - $valorResidual);
        $valorActual = max($valorResidual, $costoOriginal - $depAcumulada);
    } else {
        // Acelerada (doble declinación)
        $tasa = 2 / $vidaUtil;
        $valorActual = $costoOriginal;
        for ($i = 0; $i < $mesesTranscurridos && $i < $vidaUtil; $i++) {
            $dep = $valorActual * $tasa;
            $valorActual = max($valorResidual, $valorActual - $dep);
        }
        $depMensual = $mesesTranscurridos > 0 ? ($costoOriginal - $valorActual) / $mesesTranscurridos : 0;
    }
    
    $stmtUpdate->execute([round($valorActual, 2), round($depMensual, 2), $row['id']]);
    $actualizados++;
}

echo "Depreciación actualizada: $actualizados activos\n";
echo "Proceso completado.\n";
