const pool = require('../config/db');

class AIService {
  get apiKey() { return process.env.GEMINI_API_KEY || ''; }
  get model() { return 'gemini-2.0-flash'; }

  isEnabled() { return this.apiKey.length > 10; }

  async analyzeTicket(ticketId) {
    if (!this.isEnabled()) return null;
    const [tickets] = await pool.query('SELECT * FROM tickets WHERE id=?', [ticketId]);
    if (!tickets.length) return null;
    const t = tickets[0];

    const [articles] = await pool.query("SELECT title, LEFT(content,200) as excerpt FROM kb_articles WHERE status='published' LIMIT 8");
    const kb = articles.map(a => `- ${a.title}: ${a.excerpt}`).join('\n');

    const prompt = `Eres un asistente ITSM. Analiza este ticket. Responde SOLO JSON válido, sin backticks.
TICKET: Asunto: ${t.subject} | Descripción: ${t.description} | Prioridad: ${t.priority}
BASE CONOCIMIENTO:\n${kb}
JSON requerido: {"priority_suggestion":"low|medium|high|critical","summary":"resumen 2 líneas","suggested_response":"respuesta sugerida en español","urgency_detected":true/false,"kb_article":"título artículo relacionado o null","confidence":0.0-1.0}`;

    const text = await this.callGemini(prompt);
    if (!text) return null;
    try {
      return JSON.parse(text.replace(/```json?\s*/g, '').replace(/```/g, '').trim());
    } catch { return null; }
  }

  async suggestResponse(ticketId) {
    if (!this.isEnabled()) return null;
    const [tickets] = await pool.query('SELECT subject, description FROM tickets WHERE id=?', [ticketId]);
    if (!tickets.length) return null;
    const [comments] = await pool.query('SELECT c.comment, u.name FROM ticket_comments c JOIN users u ON c.user_id=u.id WHERE c.ticket_id=? ORDER BY c.created_at', [ticketId]);
    const history = comments.map(c => `${c.name}: ${c.comment}`).join('\n');
    const prompt = `Sugiere respuesta profesional y empática en español para este ticket de soporte.\nAsunto: ${tickets[0].subject}\nDescripción: ${tickets[0].description}\n${history ? 'Historial:\n' + history : ''}\nSolo el texto de la respuesta.`;
    return this.callGemini(prompt);
  }

  async callGemini(prompt) {
    try {
      const url = `https://generativelanguage.googleapis.com/v1beta/models/${this.model}:generateContent?key=${this.apiKey}`;
      const res = await fetch(url, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ contents: [{ parts: [{ text: prompt }] }], generationConfig: { temperature: 0.3, maxOutputTokens: 1024 } })
      });
      if (!res.ok) return null;
      const data = await res.json();
      return data.candidates?.[0]?.content?.parts?.[0]?.text || null;
    } catch (e) { console.error('Gemini error:', e.message); return null; }
  }
}

module.exports = new AIService();
