const mysql = require('mysql2/promise');
const bcrypt = require('bcryptjs');

(async () => {
  const c = await mysql.createConnection({
    host: 'localhost',
    user: 'evolucionamos_andros',
    password: 'Helpdesk2025',
    database: 'evolucionamos_andros'
  });

  console.log('🔄 Cargando datos demo...');

  // Get user IDs
  const [users] = await c.execute('SELECT id, role FROM users');
  const admin = users.find(u => u.role === 'admin');
  const agent = users.find(u => u.role === 'agent');
  const client = users.find(u => u.role === 'client');

  // Get default status ID
  const [statuses] = await c.execute('SELECT id, name FROM ticket_statuses');
  const getStatus = (name) => statuses.find(s => s.name === name)?.id || 1;

  // Get categories
  const [cats] = await c.execute('SELECT id, name FROM ticket_categories WHERE parent_id IS NULL');
  const getCat = (name) => cats.find(c => c.name.includes(name))?.id || 1;

  // Get SLA
  const [slas] = await c.execute('SELECT id, priority FROM sla_policies');
  const getSla = (p) => slas.find(s => s.priority === p)?.id || null;

  // Create demo tickets
  const tickets = [
    {
      subject: 'PC no enciende en oficina contabilidad',
      description: 'El computador de escritorio del área de contabilidad no enciende desde esta mañana. La luz del monitor parpadea pero no muestra imagen. Ya se verificó que está conectado a la corriente.',
      priority: 'high', status: 'En Progreso', category: 'Hardware',
      assigned: agent.id, created: client.id, daysAgo: 2
    },
    {
      subject: 'Error al abrir SAP - módulo de facturación',
      description: 'Al intentar ingresar al módulo de facturación de SAP muestra error "Connection timeout". Los demás módulos funcionan correctamente. Esto está bloqueando la facturación del día.',
      priority: 'critical', status: 'Abierto', category: 'Software',
      assigned: agent.id, created: client.id, daysAgo: 0
    },
    {
      subject: 'Solicitud de nuevo acceso a carpeta compartida',
      description: 'Necesito acceso a la carpeta compartida \\\\servidor\\finanzas\\reportes para poder generar los informes mensuales. Mi jefe directo es Carlos Martínez quien autorizó el acceso.',
      priority: 'medium', status: 'En Espera', category: 'Accesos',
      assigned: admin.id, created: client.id, daysAgo: 3
    },
    {
      subject: 'Impresora HP LaserJet piso 3 atascada',
      description: 'La impresora del piso 3 tiene un atasco de papel recurrente. Ya se intentó limpiar pero vuelve a atascarse después de imprimir 5-6 hojas. Afecta a todo el departamento de recursos humanos.',
      priority: 'medium', status: 'Resuelto', category: 'Impresión',
      assigned: agent.id, created: client.id, daysAgo: 5, resolved: true
    },
    {
      subject: 'VPN no conecta desde casa - teletrabajo',
      description: 'No puedo conectarme a la VPN corporativa desde mi casa. El cliente VPN muestra "Authentication failed" aunque estoy seguro que mis credenciales son correctas. Necesito acceso urgente para trabajar.',
      priority: 'high', status: 'Resuelto', category: 'Red',
      assigned: agent.id, created: client.id, daysAgo: 4, resolved: true
    },
    {
      subject: 'Instalación de Microsoft Office en laptop nueva',
      description: 'Se asignó una laptop nueva Dell Latitude 5540 al nuevo empleado Juan Pérez del departamento de ventas. Requiere instalación de Microsoft Office 365, Acrobat Reader y el cliente VPN.',
      priority: 'low', status: 'Cerrado', category: 'Software',
      assigned: agent.id, created: admin.id, daysAgo: 7, resolved: true
    },
    {
      subject: 'Correo no envía adjuntos grandes',
      description: 'Al intentar enviar correos con adjuntos mayores a 10MB el correo se queda en bandeja de salida y nunca se envía. Necesito enviar un informe de 25MB al cliente externo.',
      priority: 'medium', status: 'En Progreso', category: 'Correo',
      assigned: agent.id, created: client.id, daysAgo: 1
    },
    {
      subject: 'Monitor secundario no detectado',
      description: 'Conecté un segundo monitor HDMI a mi laptop pero no lo detecta. Ya probé otro cable HDMI y tampoco funciona. El monitor funciona bien con otro equipo.',
      priority: 'low', status: 'Abierto', category: 'Hardware',
      assigned: null, created: client.id, daysAgo: 0
    },
    {
      subject: 'WiFi lento en sala de reuniones principal',
      description: 'La conexión WiFi en la sala de reuniones del piso 2 es extremadamente lenta. Las videoconferencias se cortan constantemente. Esto afecta las reuniones con clientes.',
      priority: 'high', status: 'En Progreso', category: 'Red',
      assigned: admin.id, created: agent.id, daysAgo: 1
    },
    {
      subject: 'Resetear contraseña usuario María López',
      description: 'La usuaria María López del departamento de marketing olvidó su contraseña del dominio. Solicita reseteo urgente porque tiene una presentación en 30 minutos.',
      priority: 'critical', status: 'Cerrado', category: 'Accesos',
      assigned: admin.id, created: agent.id, daysAgo: 1, resolved: true
    }
  ];

  for (const t of tickets) {
    const createdAt = new Date(Date.now() - t.daysAgo * 86400000);
    const slaId = getSla(t.priority);
    const slaHours = t.priority === 'critical' ? 4 : t.priority === 'high' ? 8 : t.priority === 'medium' ? 24 : 48;
    const slaDue = new Date(createdAt.getTime() + slaHours * 3600000);
    const resolvedAt = t.resolved ? new Date(createdAt.getTime() + (Math.random() * 8 + 1) * 3600000) : null;
    const firstResponse = t.status !== 'Abierto' ? new Date(createdAt.getTime() + (Math.random() * 2 + 0.5) * 3600000) : null;
    const rating = t.resolved ? Math.floor(Math.random() * 2) + 4 : null; // 4 or 5

    const [result] = await c.execute(`
      INSERT INTO tickets (subject, description, priority, status_id, category_id, assigned_to, created_by,
                           sla_policy_id, sla_resolution_due, first_response_at, resolved_at,
                           satisfaction_rating, created_at, updated_at)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [t.subject, t.description, t.priority, getStatus(t.status), getCat(t.category),
       t.assigned, t.created, slaId,
       slaDue.toISOString().slice(0,19).replace('T',' '),
       firstResponse?.toISOString().slice(0,19).replace('T',' ') || null,
       resolvedAt?.toISOString().slice(0,19).replace('T',' ') || null,
       rating,
       createdAt.toISOString().slice(0,19).replace('T',' '),
       new Date().toISOString().slice(0,19).replace('T',' ')
      ]);

    const ticketId = result.insertId;

    // Add comments to some tickets
    if (t.status !== 'Abierto') {
      await c.execute(
        'INSERT INTO ticket_comments (ticket_id, user_id, comment, created_at) VALUES (?, ?, ?, ?)',
        [ticketId, agent.id, 'Estoy revisando el caso. Le mantendré informado sobre el avance.',
         new Date(createdAt.getTime() + 1800000).toISOString().slice(0,19).replace('T',' ')]);
    }
    if (t.resolved) {
      await c.execute(
        'INSERT INTO ticket_comments (ticket_id, user_id, comment, created_at) VALUES (?, ?, ?, ?)',
        [ticketId, agent.id, 'Problema resuelto. Se aplicó la solución y se verificó el funcionamiento correcto.',
         resolvedAt.toISOString().slice(0,19).replace('T',' ')]);
      await c.execute(
        'INSERT INTO ticket_comments (ticket_id, user_id, comment, created_at) VALUES (?, ?, ?, ?)',
        [ticketId, client.id, 'Confirmado, ya funciona correctamente. Gracias por la ayuda.',
         new Date(resolvedAt.getTime() + 3600000).toISOString().slice(0,19).replace('T',' ')]);
    }
  }

  // Create demo assets
  const assets = [
    { name: 'Desktop Dell OptiPlex 7090', type: 'desktop', tag: 'PC-001', serial: 'DELL7090-A1B2C3', brand: 'Dell', model: 'OptiPlex 7090', status: 'active', location: 'Piso 1 - Contabilidad', assigned: client.id },
    { name: 'Laptop Dell Latitude 5540', type: 'laptop', tag: 'LT-015', serial: 'DELL5540-X9Y8Z7', brand: 'Dell', model: 'Latitude 5540', status: 'active', location: 'Piso 2 - Ventas', assigned: agent.id },
    { name: 'Impresora HP LaserJet Pro M404', type: 'printer', tag: 'PR-003', serial: 'HPM404-D4E5F6', brand: 'HP', model: 'LaserJet Pro M404', status: 'maintenance', location: 'Piso 3 - RRHH', assigned: null },
    { name: 'Switch Cisco Catalyst 2960', type: 'network', tag: 'SW-001', serial: 'CISCO2960-G7H8I9', brand: 'Cisco', model: 'Catalyst 2960-X', status: 'active', location: 'Data Center', assigned: null },
    { name: 'Servidor Dell PowerEdge R740', type: 'server', tag: 'SRV-001', serial: 'DELLR740-J1K2L3', brand: 'Dell', model: 'PowerEdge R740', status: 'active', location: 'Data Center', assigned: null },
    { name: 'Laptop Lenovo ThinkPad T14', type: 'laptop', tag: 'LT-022', serial: 'LENT14-M4N5O6', brand: 'Lenovo', model: 'ThinkPad T14 Gen 3', status: 'active', location: 'Piso 1 - Marketing', assigned: client.id },
  ];

  for (const a of assets) {
    const [r] = await c.execute(`
      INSERT INTO assets (name, type, asset_tag, serial_number, brand, model, status, location, assigned_to, purchase_date, warranty_until)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [a.name, a.type, a.tag, a.serial, a.brand, a.model, a.status, a.location, a.assigned,
       '2024-03-15', '2027-03-15']);

    // Add maintenance for printer
    if (a.type === 'printer') {
      await c.execute(
        'INSERT INTO asset_maintenance (asset_id, type, description, cost, performed_by, performed_at) VALUES (?, ?, ?, ?, ?, NOW())',
        [r.insertId, 'repair', 'Limpieza de rodillos y cambio de kit de mantenimiento por atascos recurrentes de papel', 85.00, agent.id]);
    }
  }

  // Add audit log entries
  await c.execute("INSERT INTO audit_log (user_id, user_name, action, entity_type, entity_id, ip_address, created_at) VALUES (?, 'Administrador', 'login', 'user', ?, '192.168.1.100', NOW())", [admin.id, admin.id]);
  await c.execute("INSERT INTO audit_log (user_id, user_name, action, entity_type, entity_id, ip_address, created_at) VALUES (?, 'Agente Soporte', 'login', 'user', ?, '192.168.1.101', NOW())", [agent.id, agent.id]);

  console.log('✅ 10 tickets demo creados');
  console.log('✅ 6 activos CMDB creados');
  console.log('✅ Comentarios y auditoría agregados');
  console.log('🎉 ¡Datos demo cargados exitosamente!');

  process.exit();
})();
