const express = require('express');
const pool = require('../config/db');
const { authRequired } = require('../middlewares/auth');

module.exports = (JWT_SECRET) => {
  const router = express.Router();

  // 📋 LISTAR todos los activos
  router.get('/', authRequired(JWT_SECRET), async (req, res) => {
    try {
      const [rows] = await pool.query('SELECT * FROM activos ORDER BY codigo_inventario ASC');
      res.json(rows);
    } catch (err) {
      console.error("Error al listar activos:", err);
      res.status(500).json({ message: "Error al obtener la lista de activos" });
    }
  });

  // ➕ CREAR nuevo activo
  router.post('/', authRequired(JWT_SECRET), async (req, res) => {
    const { codigo, serial, tipo, marca, modelo, oficina, responsable, fecha } = req.body;
    
    // Validación básica de campos obligatorios
    if (!codigo || !serial || !tipo) {
      return res.status(400).json({ message: "Código, Serial y Tipo son obligatorios" });
    }

    try {
      await pool.query(
        `INSERT INTO activos (codigo_inventario, serial, tipo_equipo, marca, modelo, oficina_asignada, funcionario_responsable, fecha_ingreso) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [codigo, serial, tipo, marca, modelo, oficina, responsable, fecha]
      );
      res.json({ message: "OK" });
    } catch (err) {
      console.error("Error al insertar activo:", err);
      // Si el error es por duplicado (código 1062 en MySQL)
      if (err.code === 'ER_DUP_ENTRY') {
        return res.status(409).json({ message: "El código de inventario o serial ya existe" });
      }
      res.status(500).json({ message: "Error interno del servidor" });
    }
  });

  return router;
};