const express = require("express");
const pool = require("../config/db");
const { authRequired } = require("../middlewares/auth");

module.exports = (JWT_SECRET) => {
  const router = express.Router();

  // Definir la estructura de categorías y subcategorías de tu sistema
  const CATEGORIAS_SISTEMA = {
    'Problemas en software': [
      'Linix', 'Office', 'Windows', 'Digiturno', 
      'Bitdefender', 'WorkManager ED'
    ],
    'Problemas en hardware': [
      'Computador', 'Teclado', 'Mouse', 'Impresora', 
      'Lector de huella', 'Cámara pc', 'Audifono diadema', 
      'Video Beam', 'Sistema videoconferencia', 'Televisor',
      'Circuito cerrado de seguridad', 'Kiosco Autoconsulta',
      'Escaner', 'Puntos de conexión', 'Cables de red', 'Cables HDMI'
    ],
    'Fallas en telecomunicaciones': [
      'Celular corporativo', 'Teléfono (extensión)', 
      'Correo electrónico', 'Zoom', '3CX', 'Meet', 
      'WhatsApp', 'Planta Telefónica', 'Carpeta compartida'
    ],
    'Interrupción en el suministro de servicio público': [
      'Internet', 'Canal de datos'
    ]
  };

  // 📌 Buscar en la base de conocimiento (adaptado a tu estructura)
  router.get("/", authRequired(JWT_SECRET), async (req, res) => {
    const { search, categoria, subcategoria, limite = 10 } = req.query;
    
    try {
      // Registrar búsqueda para analytics
      if (search) {
        await pool.query(
          `INSERT INTO busquedas_ayuda (usuario_id, termino_busqueda, categoria, subcategoria, fecha) 
           VALUES (?, ?, ?, ?, NOW())`,
          [req.userId, search, categoria || null, subcategoria || null]
        );
      }

      let query = `
        SELECT 
          bc.id, 
          bc.categoria,
          bc.subcategoria,
          bc.problema, 
          bc.solucion_rapida,
          bc.solucion_detallada,
          bc.palabras_clave,
          bc.tiempo_estimado_solucion,
          bc.nivel_dificultad,
          bc.requiere_soporte,
          bc.vistas,
          bc.util_si,
          bc.util_no,
          bc.archivos_adjuntos,
          CASE 
            WHEN bc.problema LIKE ? THEN 4
            WHEN bc.subcategoria LIKE ? THEN 3
            WHEN bc.palabras_clave LIKE ? THEN 2
            WHEN bc.solucion_detallada LIKE ? THEN 1
            ELSE 0
          END as relevancia
        FROM base_conocimiento bc
        WHERE bc.activo = 1
      `;
      
      const params = [];
      
      if (search) {
        query += ` AND (bc.problema LIKE ? OR bc.palabras_clave LIKE ? OR bc.solucion_detallada LIKE ? OR bc.subcategoria LIKE ?)`;
        params.push(`%${search}%`, `%${search}%`, `%${search}%`, `%${search}%`);
      }
      
      if (categoria) {
        query += ` AND bc.categoria = ?`;
        params.push(categoria);
      }

      if (subcategoria) {
        query += ` AND bc.subcategoria = ?`;
        params.push(subcategoria);
      }
      
      query += ` ORDER BY relevancia DESC, bc.vistas DESC LIMIT ?`;
      
      // Agregar parámetros de relevancia al inicio
      const finalParams = search ? 
        [`%${search}%`, `%${search}%`, `%${search}%`, `%${search}%`, ...params, parseInt(limite)] : 
        [...params, parseInt(limite)];
      
      const [rows] = await pool.query(query, finalParams);
      
      // Incrementar contador de vistas
      if (rows.length > 0 && search) {
        const ids = rows.map(r => r.id);
        await pool.query(
          `UPDATE base_conocimiento SET vistas = vistas + 1 WHERE id IN (?)`,
          [ids]
        );
      }
      
      res.json(rows);
    } catch (err) {
      console.error("❌ Error en búsqueda:", err);
      res.status(500).json({ message: "Error en búsqueda" });
    }
  });

  // 📌 Obtener estructura de categorías del sistema
  router.get("/categorias", authRequired(JWT_SECRET), async (req, res) => {
    try {
      const [stats] = await pool.query(`
        SELECT 
          categoria,
          subcategoria,
          COUNT(*) as total_articulos,
          SUM(vistas) as total_vistas,
          AVG(CASE WHEN util_si + util_no > 0 
              THEN (util_si * 100.0) / (util_si + util_no) 
              ELSE NULL END) as efectividad
        FROM base_conocimiento
        WHERE activo = 1
        GROUP BY categoria, subcategoria
        ORDER BY categoria, subcategoria
      `);
      
      // Estructurar la respuesta con las categorías del sistema
      const resultado = {};
      for (const [cat, subcats] of Object.entries(CATEGORIAS_SISTEMA)) {
        resultado[cat] = subcats.map(subcat => {
          const stat = stats.find(s => s.categoria === cat && s.subcategoria === subcat) || {
            total_articulos: 0,
            total_vistas: 0,
            efectividad: null
          };
          return {
            subcategoria: subcat,
            ...stat
          };
        });
      }
      
      res.json(resultado);
    } catch (err) {
      console.error("❌ Error obteniendo categorías:", err);
      res.status(500).json({ message: "Error obteniendo categorías" });
    }
  });

  // 📌 Recomendaciones basadas en casos abiertos del usuario
  router.get("/recomendaciones", authRequired(JWT_SECRET), async (req, res) => {
    try {
      // Obtener casos recientes del usuario
      const [casos] = await pool.query(`
        SELECT DISTINCT 
          c.categoria,
          c.subcategoria,
          c.descripcion
        FROM casos c
        WHERE c.usuario_id = ? 
        AND c.estado IN ('abierto', 'en_progreso')
        ORDER BY c.fecha_creacion DESC
        LIMIT 5
      `, [req.userId]);

      if (casos.length === 0) {
        // Si no hay casos, mostrar soluciones populares
        const [populares] = await pool.query(`
          SELECT 
            id, 
            categoria,
            subcategoria,
            problema, 
            solucion_rapida,
            tiempo_estimado_solucion,
            nivel_dificultad
          FROM base_conocimiento
          WHERE activo = 1
          ORDER BY vistas DESC, util_si DESC
          LIMIT 5
        `);
        return res.json(populares);
      }

      // Buscar artículos relacionados con los casos abiertos
      const condiciones = casos.map(c => 
        `(bc.categoria = ? AND bc.subcategoria = ?)`
      ).join(' OR ');
      
      const params = [];
      casos.forEach(c => {
        params.push(c.categoria, c.subcategoria);
      });

      const [recomendaciones] = await pool.query(`
        SELECT DISTINCT 
          bc.id, 
          bc.categoria,
          bc.subcategoria,
          bc.problema, 
          bc.solucion_rapida,
          bc.tiempo_estimado_solucion,
          bc.nivel_dificultad,
          bc.requiere_soporte
        FROM base_conocimiento bc
        WHERE (${condiciones})
        AND bc.activo = 1
        ORDER BY bc.util_si DESC, bc.vistas DESC
        LIMIT 10
      `, params);

      res.json(recomendaciones);
    } catch (err) {
      console.error("❌ Error obteniendo recomendaciones:", err);
      res.status(500).json({ message: "Error obteniendo recomendaciones" });
    }
  });

  // 📌 Obtener soluciones por subcategoría específica
  router.get("/soluciones/:categoria/:subcategoria", authRequired(JWT_SECRET), async (req, res) => {
    const { categoria, subcategoria } = req.params;
    
    try {
      const [soluciones] = await pool.query(`
        SELECT 
          id,
          problema,
          solucion_rapida,
          solucion_detallada,
          tiempo_estimado_solucion,
          nivel_dificultad,
          requiere_soporte,
          archivos_adjuntos,
          vistas,
          util_si,
          util_no
        FROM base_conocimiento
        WHERE categoria = ? AND subcategoria = ? AND activo = 1
        ORDER BY 
          CASE nivel_dificultad 
            WHEN 'facil' THEN 1 
            WHEN 'medio' THEN 2 
            WHEN 'dificil' THEN 3 
          END,
          vistas DESC
      `, [categoria, subcategoria]);

      res.json(soluciones);
    } catch (err) {
      console.error("❌ Error obteniendo soluciones:", err);
      res.status(500).json({ message: "Error obteniendo soluciones" });
    }
  });

  // 📌 Obtener detalle completo de una solución
  router.get("/solucion/:id", authRequired(JWT_SECRET), async (req, res) => {
    const { id } = req.params;
    
    try {
      const [rows] = await pool.query(`
        SELECT 
          bc.*,
          u.nombre as autor_nombre
        FROM base_conocimiento bc
        LEFT JOIN usuarios u ON bc.creado_por = u.id
        WHERE bc.id = ? AND bc.activo = 1
      `, [id]);

      if (rows.length === 0) {
        return res.status(404).json({ message: "Solución no encontrada" });
      }

      // Incrementar vistas
      await pool.query(
        `UPDATE base_conocimiento SET vistas = vistas + 1 WHERE id = ?`,
        [id]
      );

      const solucion = rows[0];

      // Obtener soluciones relacionadas (misma subcategoría)
      const [relacionadas] = await pool.query(`
        SELECT id, problema, nivel_dificultad, tiempo_estimado_solucion
        FROM base_conocimiento
        WHERE categoria = ? 
        AND subcategoria = ? 
        AND id != ?
        AND activo = 1
        ORDER BY vistas DESC
        LIMIT 3
      `, [solucion.categoria, solucion.subcategoria, id]);

      // Obtener historial de efectividad
      const [efectividad] = await pool.query(`
        SELECT 
          COUNT(CASE WHEN util = 1 THEN 1 END) as resolvio_problema,
          COUNT(CASE WHEN util = 0 THEN 1 END) as no_resolvio,
          COUNT(*) as total_calificaciones
        FROM calificaciones_ayuda
        WHERE articulo_id = ?
      `, [id]);

      res.json({
        solucion,
        relacionadas,
        efectividad: efectividad[0]
      });
    } catch (err) {
      console.error("❌ Error obteniendo solución:", err);
      res.status(500).json({ message: "Error obteniendo solución" });
    }
  });

  // 📌 Calificar si una solución funcionó
  router.post("/solucion/:id/calificar", authRequired(JWT_SECRET), async (req, res) => {
    const { id } = req.params;
    const { funciono, comentario, tiempo_resolucion } = req.body;
    
    try {
      // Verificar si ya calificó
      const [existe] = await pool.query(`
        SELECT id FROM calificaciones_ayuda 
        WHERE usuario_id = ? AND articulo_id = ?
      `, [req.userId, id]);

      if (existe.length > 0) {
        // Actualizar calificación existente
        await pool.query(`
          UPDATE calificaciones_ayuda 
          SET util = ?, comentario = ?, tiempo_resolucion = ?, fecha = NOW()
          WHERE usuario_id = ? AND articulo_id = ?
        `, [funciono, comentario, tiempo_resolucion, req.userId, id]);
      } else {
        // Nueva calificación
        await pool.query(`
          INSERT INTO calificaciones_ayuda 
          (usuario_id, articulo_id, util, comentario, tiempo_resolucion, fecha)
          VALUES (?, ?, ?, ?, ?, NOW())
        `, [req.userId, id, funciono, comentario, tiempo_resolucion]);
      }

      // Actualizar contadores
      const campo = funciono ? 'util_si' : 'util_no';
      await pool.query(
        `UPDATE base_conocimiento SET ${campo} = ${campo} + 1 WHERE id = ?`,
        [id]
      );

      res.json({ message: "Gracias por tu retroalimentación" });
    } catch (err) {
      console.error("❌ Error al calificar:", err);
      res.status(500).json({ message: "Error al calificar" });
    }
  });

  // 📌 Crear ticket si la solución no funcionó
  router.post("/crear-ticket", authRequired(JWT_SECRET), async (req, res) => {
    const { 
      categoria, 
      subcategoria, 
      descripcion, 
      solucion_intentada_id,
      urgencia 
    } = req.body;
    
    try {
      // Determinar prioridad basada en la urgencia y categoría
      let prioridad = 'media';
      if (urgencia === 'alta' || categoria === 'Interrupción en el suministro de servicio público') {
        prioridad = 'alta';
      } else if (categoria === 'Problemas en software' && ['Linix', 'Digiturno', 'WorkManager ED'].includes(subcategoria)) {
        prioridad = 'alta';
      }

      const [result] = await pool.query(`
        INSERT INTO casos (
          usuario_id, 
          categoria,
          subcategoria,
          titulo, 
          descripcion, 
          prioridad,
          estado,
          fecha_creacion,
          origen,
          solucion_intentada_id
        ) VALUES (?, ?, ?, ?, ?, ?, 'abierto', NOW(), 'mesa_ayuda', ?)
      `, [
        req.userId, 
        categoria, 
        subcategoria,
        `${categoria} - ${subcategoria}`,
        descripcion, 
        prioridad,
        solucion_intentada_id
      ]);

      // Registrar en analytics
      await pool.query(`
        INSERT INTO tickets_desde_ayuda (caso_id, solucion_id, fecha_creacion)
        VALUES (?, ?, NOW())
      `, [result.insertId, solucion_intentada_id]);

      res.json({ 
        message: "Ticket creado exitosamente",
        ticketId: result.insertId,
        prioridad: prioridad
      });
    } catch (err) {
      console.error("❌ Error creando ticket:", err);
      res.status(500).json({ message: "Error creando ticket" });
    }
  });

  // 📌 Obtener soluciones rápidas (sin requerir soporte)
  router.get("/soluciones-rapidas", authRequired(JWT_SECRET), async (req, res) => {
    const { categoria } = req.query;
    
    try {
      let query = `
        SELECT 
          id,
          categoria,
          subcategoria,
          problema,
          solucion_rapida,
          tiempo_estimado_solucion,
          nivel_dificultad
        FROM base_conocimiento
        WHERE activo = 1 
        AND requiere_soporte = 0
        AND nivel_dificultad IN ('facil', 'medio')
      `;
      
      const params = [];
      if (categoria) {
        query += ` AND categoria = ?`;
        params.push(categoria);
      }
      
      query += ` ORDER BY vistas DESC, util_si DESC LIMIT 10`;
      
      const [soluciones] = await pool.query(query, params);
      res.json(soluciones);
    } catch (err) {
      console.error("❌ Error obteniendo soluciones rápidas:", err);
      res.status(500).json({ message: "Error obteniendo soluciones rápidas" });
    }
  });

  // 📌 Buscar problemas similares resueltos
  router.get("/problemas-similares", authRequired(JWT_SECRET), async (req, res) => {
    const { descripcion, categoria, subcategoria } = req.query;
    
    try {
      // Buscar casos similares ya resueltos
      const [casos_resueltos] = await pool.query(`
        SELECT 
          c.id as caso_id,
          c.titulo,
          c.descripcion,
          c.solucion_final,
          c.fecha_resolucion,
          c.categoria,
          c.subcategoria,
          u.nombre as resuelto_por
        FROM casos c
        LEFT JOIN usuarios u ON c.tecnico_asignado = u.id
        WHERE c.estado = 'cerrado'
        AND c.solucion_final IS NOT NULL
        AND c.categoria = ?
        AND c.subcategoria = ?
        AND (c.descripcion LIKE ? OR c.titulo LIKE ?)
        ORDER BY c.fecha_resolucion DESC
        LIMIT 5
      `, [categoria, subcategoria, `%${descripcion}%`, `%${descripcion}%`]);

      res.json(casos_resueltos);
    } catch (err) {
      console.error("❌ Error buscando problemas similares:", err);
      res.status(500).json({ message: "Error buscando problemas similares" });
    }
  });

  // 📌 Analytics para administradores (adaptado a tu sistema)
  router.get("/analytics", authRequired(JWT_SECRET), async (req, res) => {
    try {
      // Verificar si es admin o soporte
      const [user] = await pool.query(
        "SELECT rol FROM usuarios WHERE id = ?",
        [req.userId]
      );

      if (user[0]?.rol !== 'admin' && user[0]?.rol !== 'soporte') {
        return res.status(403).json({ message: "No autorizado" });
      }

      // Problemas más comunes por categoría
      const [problemas_frecuentes] = await pool.query(`
        SELECT 
          categoria,
          subcategoria,
          COUNT(*) as total_busquedas
        FROM busquedas_ayuda
        WHERE fecha >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY categoria, subcategoria
        ORDER BY total_busquedas DESC
        LIMIT 10
      `);

      // Efectividad de soluciones
      const [efectividad] = await pool.query(`
        SELECT 
          bc.categoria,
          bc.subcategoria,
          COUNT(DISTINCT bc.id) as total_soluciones,
          AVG(CASE WHEN bc.util_si + bc.util_no > 0 
              THEN (bc.util_si * 100.0) / (bc.util_si + bc.util_no) 
              ELSE NULL END) as porcentaje_efectividad,
          SUM(bc.vistas) as total_vistas
        FROM base_conocimiento bc
        WHERE bc.activo = 1
        GROUP BY bc.categoria, bc.subcategoria
        ORDER BY bc.categoria, bc.subcategoria
      `);

      // Tickets generados desde mesa de ayuda
      const [tickets_generados] = await pool.query(`
        SELECT 
          DATE(fecha_creacion) as fecha,
          COUNT(*) as cantidad
        FROM casos
        WHERE origen = 'mesa_ayuda'
        AND fecha_creacion >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(fecha_creacion)
        ORDER BY fecha DESC
      `);

      // Soluciones que necesitan mejora (baja efectividad)
      const [necesitan_mejora] = await pool.query(`
        SELECT 
          id,
          categoria,
          subcategoria,
          problema,
          util_si,
          util_no,
          ROUND((util_no * 100.0) / (util_si + util_no), 1) as porcentaje_fallo
        FROM base_conocimiento
        WHERE (util_si + util_no) >= 5
        AND (util_no * 100.0) / (util_si + util_no) > 30
        ORDER BY porcentaje_fallo DESC
        LIMIT 10
      `);

      res.json({
        problemas_frecuentes,
        efectividad,
        tickets_generados,
        necesitan_mejora
      });
    } catch (err) {
      console.error("❌ Error obteniendo analytics:", err);
      res.status(500).json({ message: "Error obteniendo analytics" });
    }
  });

  return router;
};