// backend/routes/reportes.routes.js
const express = require('express');
const pool = require('../config/db');
const { authRequired } = require('../middlewares/auth');
const mailer = require('../config/email');
const multer = require('multer'); 
const path = require('path');

// 📂 CONFIGURACIÓN DE MULTER (Manejo de archivos)
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, 'public/uploads/'); 
  },
  filename: function (req, file, cb) {
    // Nombre único para evitar sobrescribir fotos: fecha + aleatorio
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    cb(null, uniqueSuffix + path.extname(file.originalname));
  }
});

const upload = multer({ 
  storage: storage,
  limits: { fileSize: 10 * 1024 * 1024 } // Límite de 10MB por imagen
});

module.exports = (JWT_SECRET) => {
  const router = express.Router();

  // ---------- Helpers para el correo ----------
  const escapeHtml = (str) =>
    String(str ?? '')
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#39;');

  const nl2br = (str) => escapeHtml(str).replace(/\r?\n/g, '<br>');

  const DASHBOARD_URL = process.env.ADMIN_DASHBOARD_URL || 'https://david.fedef.com/dashboard_admin.html';

  // ==========================================
  // POST → Registrar caso (CON IMAGEN)
  // ==========================================
  router.post('/', authRequired(JWT_SECRET), upload.single('imagen'), async (req, res) => {
    try {
      const { fecha, oficina, tipoEvento, detalle } = req.body;
      const imagenNombre = req.file ? req.file.filename : null; 

      const pick = (...vals) => {
        for (const v of vals) {
          if (typeof v === 'string' && v.trim()) return v.trim();
        }
        return '';
      };

      const equipoAfectado = pick(
        req.body.equipoAfectado,
        req.body.equipo_afectado,
        req.body['equipo-afectado'],
        req.body.equipo
      );

      const funcionario = req.user?.nombre;

      if (!fecha || !tipoEvento || !detalle || !equipoAfectado) {
        return res.status(400).json({ message: 'Campos obligatorios faltantes' });
      }

      // ✅ INSERTANDO EN LA TABLA CON LA NUEVA COLUMNA IMAGEN
      const [result] = await pool.query(
        `INSERT INTO reportes_soporte
         (fecha, funcionario, oficina, equipo_afectado, tipo_evento, detalle, estado, imagen)
         VALUES (?, ?, ?, ?, ?, ?, 'En proceso', ?)`,
        [fecha, funcionario, oficina || null, equipoAfectado, tipoEvento, detalle.trim(), imagenNombre]
      );

      res.json({ message: '✅ Caso reportado con éxito', id: result.insertId });

      // —— ENVÍO DE NOTIFICACIÓN (DISEÑO EXTENDIDO)
      try {
        const [admins] = await pool.query(`SELECT email FROM usuarios WHERE tipo = 'admin' AND activo = 1`);
        if (admins.length > 0) {
          const bccList = admins.map(a => a.email).filter(Boolean);
          const S = {
            id: result.insertId,
            fecha: escapeHtml(fecha),
            funcionario: escapeHtml(funcionario),
            oficina: escapeHtml(oficina || 'No especificada'),
            equipo: escapeHtml(equipoAfectado),
            tipo: escapeHtml(tipoEvento),
            detalleHtml: nl2br(detalle.trim()),
            hayImagen: imagenNombre ? 'SÍ (Imagen adjunta)' : 'No'
          };

          await mailer.sendMail({
            from: process.env.MAIL_FROM || process.env.EMAIL_USER,
            bcc: bccList,
            subject: `📢 Nuevo reporte de soporte (#${S.id}) - FEDEF`,
            html: `
            <table role="presentation" width="100%" cellspacing="0" cellpadding="0" border="0" style="background:#f3f4f6;padding:24px 12px;margin:0;font-family:Arial,sans-serif">
              <tr>
                <td align="center">
                  <table role="presentation" width="640" style="width:640px;background:#ffffff;border-radius:12px;border:1px solid #e5e7eb;">
                    <tr>
                      <td style="background:#16a34a;padding:30px;color:#ffffff;text-align:center;border-radius:12px 12px 0 0">
                        <div style="font-size:24px;font-weight:700">Nuevo Reporte Recibido</div>
                        <div style="margin-top:10px;background:rgba(255,255,255,0.2);display:inline-block;padding:5px 15px;border-radius:20px">Ticket #${S.id}</div>
                      </td>
                    </tr>
                    <tr>
                      <td style="padding:30px">
                        <p style="color:#4b5563;font-size:16px">Hola Administrador, se ha generado un nuevo requerimiento:</p>
                        <table width="100%" style="border-collapse:collapse;margin-top:20px">
                          <tr><td style="padding:10px;border-bottom:1px solid #f3f4f6;font-weight:bold">Funcionario:</td><td style="padding:10px;border-bottom:1px solid #f3f4f6">${S.funcionario}</td></tr>
                          <tr><td style="padding:10px;border-bottom:1px solid #f3f4f6;font-weight:bold">Oficina:</td><td style="padding:10px;border-bottom:1px solid #f3f4f6">${S.oficina}</td></tr>
                          <tr><td style="padding:10px;border-bottom:1px solid #f3f4f6;font-weight:bold">Afectación:</td><td style="padding:10px;border-bottom:1px solid #f3f4f6">${S.equipo}</td></tr>
                          <tr><td style="padding:10px;border-bottom:1px solid #f3f4f6;font-weight:bold">¿Tiene foto?:</td><td style="padding:10px;border-bottom:1px solid #f3f4f6">${S.hayImagen}</td></tr>
                        </table>
                        <div style="margin-top:25px;padding:20px;background:#f9fafb;border-radius:10px;border-left:4px solid #16a34a">
                          <div style="font-weight:bold;margin-bottom:10px">Descripción del problema:</div>
                          <div style="color:#374151;line-height:1.5">${S.detalleHtml}</div>
                        </div>
                        <div style="text-align:center;margin-top:35px">
                          <a href="${DASHBOARD_URL}" style="background:#16a34a;color:#ffffff;padding:15px 30px;text-decoration:none;border-radius:8px;font-weight:bold;display:inline-block">Gestionar en el Dashboard</a>
                        </div>
                      </td>
                    </tr>
                  </table>
                </td>
              </tr>
            </table>`
          });
        }
      } catch (mailErr) { console.error('Error enviando mail:', mailErr); }
    } catch (err) {
      console.error(err);
      res.status(500).json({ message: 'Error interno del servidor' });
    }
  });

  // ==========================================
  // GET → Listar casos del usuario autenticado
  // ==========================================
  router.get('/mis-casos', authRequired(JWT_SECRET), async (req, res) => {
    try {
      const funcionario = req.user.nombre;
      const [rows] = await pool.query(
        `SELECT id, fecha, oficina, equipo_afectado, tipo_evento, detalle, estado, fecha_creacion, fecha_finalizacion, finalizado_por, imagen,
         CASE WHEN fecha_finalizacion IS NOT NULL THEN TIMESTAMPDIFF(SECOND, fecha_creacion, fecha_finalizacion) ELSE NULL END AS duracion_segundos
         FROM reportes_soporte WHERE funcionario = ? ORDER BY fecha_creacion DESC, id DESC`, [funcionario]
      );
      res.json(rows);
    } catch (err) {
      console.error(err);
      res.status(500).json({ message: 'Error al obtener tus casos' });
    }
  });

  // ==========================================
  // GET → Listar todos los reportes (Admin)
  // ==========================================
  router.get('/todos', authRequired(JWT_SECRET), async (req, res) => {
    try {
      if (req.user.rol !== 'admin') return res.status(403).json({ message: 'No autorizado' });
      const { estado } = req.query;
      let where = '';
      const params = [];

      if (estado && estado !== 'Todos') {
        where = 'WHERE estado = ?';
        params.push(estado);
      }

      const [rows] = await pool.query(
        `SELECT id, fecha, funcionario, oficina, equipo_afectado, tipo_evento, detalle, estado, fecha_creacion, fecha_finalizacion, finalizado_por, imagen,
         CASE WHEN fecha_finalizacion IS NOT NULL THEN TIMESTAMPDIFF(SECOND, fecha_creacion, fecha_finalizacion) ELSE NULL END AS duracion_segundos
         FROM reportes_soporte ${where} ORDER BY fecha_creacion DESC`, params
      );
      res.json(rows);
    } catch (err) {
      console.error(err);
      res.status(500).json({ message: 'Error al obtener reportes' });
    }
  });

  // ==========================================
  // GET → ESTADÍSTICAS (Versión Completa)
  // ==========================================
  router.get('/estadisticas', authRequired(JWT_SECRET), async (req, res) => {
    try {
      // 1. Por fecha
      const [porFecha] = await pool.query(`SELECT DATE(fecha_creacion) AS fecha, COUNT(*) AS total FROM reportes_soporte WHERE fecha_creacion >= (CURRENT_DATE() - INTERVAL 30 DAY) GROUP BY DATE(fecha_creacion) ORDER BY fecha ASC`);
      // 2. Por estado
      const [porEstado] = await pool.query(`SELECT estado, COUNT(*) AS total FROM reportes_soporte GROUP BY estado`);
      // 3. Por tipo
      const [porTipo] = await pool.query(`SELECT tipo_evento, COUNT(*) AS total FROM reportes_soporte GROUP BY tipo_evento`);
      // 4. Por oficina
      const [porOficina] = await pool.query(`SELECT IFNULL(oficina, 'Sin oficina') AS oficina, COUNT(*) AS total FROM reportes_soporte GROUP BY oficina`);
      // 5. Por admin
      const [porAdmin] = await pool.query(`SELECT IFNULL(finalizado_por, '—') AS finalizado_por, COUNT(*) AS total FROM reportes_soporte WHERE finalizado_por IS NOT NULL GROUP BY finalizado_por`);
      // 6. Promedio global
      const [promRow] = await pool.query(`SELECT AVG(TIMESTAMPDIFF(SECOND, fecha_creacion, fecha_finalizacion)) AS promedio FROM reportes_soporte WHERE estado = 'Finalizado' AND fecha_finalizacion IS NOT NULL`);
      // 7. Tiempo promedio por día
      const [tiempoPromedioPorDia] = await pool.query(`SELECT DATE(fecha_finalizacion) AS fecha, AVG(TIMESTAMPDIFF(SECOND, fecha_creacion, fecha_finalizacion)) AS avg_segundos FROM reportes_soporte WHERE estado = 'Finalizado' AND fecha_finalizacion >= (CURRENT_DATE() - INTERVAL 30 DAY) GROUP BY DATE(fecha_finalizacion) ORDER BY fecha ASC`);

      res.json({
        porFecha, porEstado, porTipo, porOficina, porAdmin,
        promedioGlobalSegundos: promRow[0]?.promedio || null,
        tiempoPromedioPorDia
      });
    } catch (err) {
      console.error('Error estadísticas:', err);
      res.status(500).json({ message: 'Error al generar estadísticas' });
    }
  });

  // ==========================================
  // PUT → Cambiar estado del reporte
  // ==========================================
  router.put('/:id/estado', authRequired(JWT_SECRET), async (req, res) => {
    try {
      const { id } = req.params;
      const { nuevoEstado } = req.body;
      const admin = req.user.nombre;

      if (!nuevoEstado) return res.status(400).json({ message: 'Estado requerido' });

      if (nuevoEstado === 'Finalizado') {
        await pool.query(
          `UPDATE reportes_soporte SET estado = ?, finalizado_por = ?, fecha_finalizacion = NOW() WHERE id = ?`,
          [nuevoEstado, admin, id]
        );
      } else {
        await pool.query(
          `UPDATE reportes_soporte SET estado = ?, finalizado_por = NULL, fecha_finalizacion = NULL WHERE id = ?`,
          [nuevoEstado, id]
        );
      }

      res.json({ message: '✅ Estado actualizado correctamente', id, nuevoEstado });
    } catch (err) {
      console.error(err);
      res.status(500).json({ message: 'Error interno del servidor' });
    }
  });

  return router;
};