require('dotenv').config();
// 🔥 PARCHE DE SEGURIDAD GLOBAL: Evita que el servidor se caiga por certificados de correo (SMTP)
process.env.NODE_TLS_REJECT_UNAUTHORIZED = '0';

const express = require('express');
const cors = require('cors');
const path = require('path');
const jwt = require('jsonwebtoken');
const pool = require('./config/db');
const { authRequired } = require('./middlewares/auth');

// 🛡️ ESCUDO PROTECTOR: Atrapa errores inesperados para que el servidor no se apague
process.on('unhandledRejection', (reason, promise) => {
    console.error('⚠️ Rechazo no manejado:', reason);
});
process.on('uncaughtException', (err) => {
    console.error('❌ Excepción crítica:', err.message);
});

const app = express();
const PORT = process.env.PORT || 4000;
const JWT_SECRET = process.env.JWT_SECRET || 'dev_secret_change_me';
const JWT_EXPIRES_IN = process.env.JWT_EXPIRES_IN || '30m';

app.use(cors());
app.use(express.json());

// Rutas
const soporteRoutes = require('./routes/soporte.routes');
const reportesRoutes = require('./routes/reportes.routes');
const ayudaRoutes = require("./routes/ayuda.routes")(JWT_SECRET);
const helpdeskRoutes = require("./routes/helpdesk.routes");
const recomendacionesRoutes = require("./routes/recomendaciones.routes");
// Dentro de backend/server.js, junto a las otras rutas:
// En backend/server.js
const activosRoutes = require('./routes/activos.routes');
app.use('/api/activos', activosRoutes(JWT_SECRET));
const conocimientoRoutes = require('./routes/conocimiento.routes');
app.use('/api/conocimiento', conocimientoRoutes(JWT_SECRET));
// 1. Importa la ruta (al principio con los otros requires)
const usuariosRoutes = require("./routes/usuarios.routes"); // 1. Importar
// ... (después de las otras rutas de API)
app.use("/api/usuarios", usuariosRoutes(JWT_SECRET));      // 2. Registrar

app.use("/api/soportes", soporteRoutes(JWT_SECRET));
app.use("/api/reportes", reportesRoutes(JWT_SECRET));
app.use("/api/ayuda", ayudaRoutes);
app.use("/api/helpdesk", helpdeskRoutes(JWT_SECRET));
app.use("/api/recomendaciones", recomendacionesRoutes);

app.use(express.static(path.join(__dirname, '..', 'public')));

// LOGIN CORREGIDO (Para evitar el "undefined")
app.post('/api/auth/login', async (req, res) => {
  try {
    const { nombre, password } = req.body;
    const [rows] = await pool.query('SELECT * FROM usuarios WHERE nombre = ? AND activo = 1', [nombre]);
    
    if (rows.length === 0 || password !== rows[0].password) {
      return res.status(401).json({ message: 'Credenciales inválidas' });
    }

    const user = rows[0];
    const token = jwt.sign({ sub: user.id, nombre: user.nombre, rol: user.tipo }, JWT_SECRET, { expiresIn: JWT_EXPIRES_IN });

    console.log(`✅ Sesión iniciada para: ${user.nombre}`);
    
    // Enviamos "user" y "nombre" para que el frontend lo encuentre sí o sí
    res.json({ 
      token, 
      user: user.nombre, 
      nombre: user.nombre, 
      rol: user.tipo 
    });
  } catch (err) {
    console.error('❌ Error en Login:', err.message);
    res.status(500).json({ message: 'Error interno' });
  }
});

app.get('/api/dashboard', authRequired(JWT_SECRET), (req, res) => {
  res.json({ ok: true, user: req.user });
});

app.get('*', (req, res) => {
  res.sendFile(path.join(__dirname, '..', 'public', 'index.html'));
});

// Verificación de Correo segura
const mailer = require('./config/email');
mailer.verify()
  .then(() => console.log('✅ SMTP listo para FEDEF'))
  .catch(err => console.log('⚠️ Aviso: Correo no disponible, pero el servidor sigue vivo.'));

const HOST = '0.0.0.0';
app.listen(PORT, HOST, () => {
    console.log(`🚀 Servidor FEDEF a toda marcha en el puerto ${PORT}`);
});