import { Chart } from 'chart.js';
import { fetchWithAuth, clearToken } from '/js/auth.js';
/* ==========================
   Función helper: agrupar por campo
========================== */
function agruparPor(arr, campo) {
  return arr.reduce((acc, item) => {
    const key = item[campo] || 'Sin dato';
    acc[key] = (acc[key] || 0) + 1;
    return acc;
  }, {});
}

/* ==========================
   1. Cargar datos desde backend
========================== */
async function cargarDatos() {
  try {
    const res = await fetchWithAuth('/api/reportes/todos');
    if (!res.ok) throw new Error('Error al obtener datos');
    return await res.json();
  } catch (err) {
    console.error('❌ Error cargando reportes:', err);
    return [];
  }
}

/* ==========================
   2. Inicializar gráficas
========================== */
async function initCharts() {
  const data = await cargarDatos();

  if (data.length === 0) {
    console.warn('⚠️ No hay datos para graficar');
    return;
  }

  // --- Reportes en el tiempo (por fecha de creación)
  const porFecha = agruparPor(
    data.map(r => new Date(r.fecha_creacion).toLocaleDateString('es-CO')),
    null
  );
  const fechas = Object.keys(porFecha);
  const valoresFechas = Object.values(porFecha);

  new Chart(document.getElementById('chartLine'), {
    type: 'line',
    data: {
      labels: fechas,
      datasets: [{
        label: 'Casos reportados',
        data: valoresFechas,
        borderColor: '#007A33',
        backgroundColor: 'rgba(0, 122, 51, 0.2)',
        tension: 0.3,
        fill: true
      }]
    },
    options: {
      responsive: true,
      plugins: { legend: { display: false } }
    }
  });

  // --- Estados (Finalizado vs En proceso)
  const estados = agruparPor(data, 'estado');
  new Chart(document.getElementById('chartDoughnut'), {
    type: 'doughnut',
    data: {
      labels: Object.keys(estados),
      datasets: [{
        data: Object.values(estados),
        backgroundColor: ['#16a34a', '#f59e0b', '#ef4444']
      }]
    },
    options: { responsive: true }
  });

  // --- Casos por tipo de evento
  const porTipo = agruparPor(data, 'tipo_evento');
  new Chart(document.getElementById('chartBarTipo'), {
    type: 'bar',
    data: {
      labels: Object.keys(porTipo),
      datasets: [{
        label: 'Cantidad de casos',
        data: Object.values(porTipo),
        backgroundColor: '#3b82f6'
      }]
    },
    options: { responsive: true }
  });

  // --- Casos por oficina
  const porOficina = agruparPor(data, 'oficina');
  new Chart(document.getElementById('chartBarOficina'), {
    type: 'bar',
    data: {
      labels: Object.keys(porOficina),
      datasets: [{
        label: 'Cantidad de casos',
        data: Object.values(porOficina),
        backgroundColor: '#8b5cf6'
      }]
    },
    options: { responsive: true }
  });

  // --- Casos cerrados por administrador
  const porAdmin = agruparPor(
    data.filter(r => r.finalizado_por), // solo si fue cerrado
    'finalizado_por'
  );
  new Chart(document.getElementById('chartBarAdmins'), {
    type: 'bar',
    data: {
      labels: Object.keys(porAdmin),
      datasets: [{
        label: 'Casos cerrados',
        data: Object.values(porAdmin),
        backgroundColor: '#f97316'
      }]
    },
    options: { responsive: true }
  });
}

/* ==========================
   3. Logout
========================== */
document.getElementById('logoutBtn').addEventListener('click', () => {
  clearToken();
  window.location.href = '/';
});

/* ==========================
   4. Ejecutar
========================== */
document.addEventListener('DOMContentLoaded', initCharts);
