// /js/admin_estadisticas.js
import { fetchWithAuth, clearToken } from '/js/auth.js';

const ctxLine    = document.getElementById('chartPorFecha');
const ctxEstado  = document.getElementById('chartPorEstado');
const ctxTipo    = document.getElementById('chartPorTipo');
const ctxOficina = document.getElementById('chartPorOficina');
const ctxAdmin   = document.getElementById('chartPorAdmin');

// ===== helpers UI =====
const paintMsgInChart = (canvasEl, msg) => {
  if (!canvasEl) return;
  const cont = canvasEl.closest('.chart-container');
  if (cont) {
    cont.innerHTML = `<div class="text-center text-gray-400 py-10">${msg}</div>`;
  }
};

const humanizarSegundos = (total) => {
  if (total == null) return '—';
  const s = Number(total);
  const d = Math.floor(s / 86400);
  const h = Math.floor((s % 86400) / 3600);
  const m = Math.floor((s % 3600) / 60);
  const sec = s % 60;
  const parts = [];
  if (d) parts.push(`${d}d`);
  if (h) parts.push(`${h}h`);
  if (m) parts.push(`${m}m`);
  if (!d && !h && !m) parts.push(`${sec}s`);
  return parts.join(' ');
};

// evitar superposición de charts al recargar
const makeChart = (key, ctx, config) => {
  if (!ctx) return;
  window.__charts = window.__charts || {};
  if (window.__charts[key]) {
    try { window.__charts[key].destroy(); } catch {}
  }
  window.__charts[key] = new Chart(ctx, config);
};

async function cargarEstadisticas() {
  try {
    const res = await fetchWithAuth('/api/reportes/estadisticas', { cache: 'no-store' });
    if (!res.ok) {
      // intenta leer el cuerpo de error (si existe) para loguear mejor
      let bodyText = '';
      try { bodyText = await res.text(); } catch {}
      console.error(`[estadisticas] HTTP ${res.status} →`, bodyText || '(sin body)');
      throw new Error(`❌ Error al obtener estadísticas (HTTP ${res.status})`);
    }

    const stats = await res.json();
    console.log('[estadisticas] payload:', stats);

    // ===== KPI: promedio global =====
    const kpiEl = document.getElementById('avgResolucionKpi');
    if (kpiEl) {
      kpiEl.textContent = (stats?.promedioGlobalSegundos != null)
        ? humanizarSegundos(stats.promedioGlobalSegundos)
        : '—';
    }

    // ===== Gráfica: tiempo promedio por día (minutos) =====
    const ctxTiempo = document.getElementById('chartTiempoPromedio');
    if (ctxTiempo) {
      const serie = Array.isArray(stats?.tiempoPromedioPorDia) ? stats.tiempoPromedioPorDia : [];
      if (serie.length === 0) {
        paintMsgInChart(ctxTiempo, 'No hay datos suficientes (finaliza algunos casos para ver el promedio).');
      } else {
        const labels = serie.map(r => r.fecha);
        const dataMin = serie.map(r => Number((r.avg_segundos / 60).toFixed(1)));
        makeChart('tiempoPromedio', ctxTiempo, {
          type: 'line',
          data: {
            labels,
            datasets: [{
              label: 'Promedio (minutos)',
              data: dataMin,
              borderColor: '#007A33',
              backgroundColor: 'rgba(0, 122, 51, 0.15)',
              borderWidth: 2,
              tension: 0.25,
              pointRadius: 3,
              fill: true
            }]
          },
          options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
              y: { beginAtZero: true, title: { display: true, text: 'Minutos' } },
              x: { title: { display: true, text: 'Fecha de finalización' } }
            },
            plugins: {
              tooltip: { callbacks: { label: (ctx) => `${ctx.parsed.y} min` } },
              legend: { display: true }
            }
          }
        });
      }
    }

    // ===== 1) Reportes por fecha =====
    const porFecha = Array.isArray(stats?.porFecha) ? stats.porFecha : [];
    if (ctxLine) {
      if (porFecha.length === 0) {
        paintMsgInChart(ctxLine, 'Sin datos para "Reportes por día".');
      } else {
        makeChart('porFecha', ctxLine, {
          type: 'line',
          data: {
            labels: porFecha.map(f => f.fecha),
            datasets: [{
              label: 'Reportes',
              data: porFecha.map(f => f.total),
              borderColor: '#007A33',
              backgroundColor: 'rgba(0, 122, 51, 0.3)',
              fill: true,
              tension: 0.3
            }]
          }
        });
      }
    }

    // ===== 2) Estados =====
    const porEstado = Array.isArray(stats?.porEstado) ? stats.porEstado : [];
    if (ctxEstado) {
      if (porEstado.length === 0) {
        paintMsgInChart(ctxEstado, 'Sin datos de estados.');
      } else {
        makeChart('porEstado', ctxEstado, {
          type: 'doughnut',
          data: {
            labels: porEstado.map(e => e.estado),
            datasets: [{
              label: 'Estados',
              data: porEstado.map(e => e.total),
              backgroundColor: ['#16a34a', '#f59e0b', '#ef4444']
            }]
          }
        });
      }
    }

    // ===== 3) Tipos de evento =====
    const porTipo = Array.isArray(stats?.porTipo) ? stats.porTipo : [];
    if (ctxTipo) {
      if (porTipo.length === 0) {
        paintMsgInChart(ctxTipo, 'Sin datos de tipos de evento.');
      } else {
        makeChart('porTipo', ctxTipo, {
          type: 'bar',
          data: {
            labels: porTipo.map(t => t.tipo_evento),
            datasets: [{
              label: 'Cantidad',
              data: porTipo.map(t => t.total),
              backgroundColor: '#3b82f6'
            }]
          },
          options: { indexAxis: 'y' }
        });
      }
    }

    // ===== 4) Oficinas =====
    const porOficina = Array.isArray(stats?.porOficina) ? stats.porOficina : [];
    if (ctxOficina) {
      if (porOficina.length === 0) {
        paintMsgInChart(ctxOficina, 'Sin datos por oficina.');
      } else {
        makeChart('porOficina', ctxOficina, {
          type: 'bar',
          data: {
            labels: porOficina.map(o => o.oficina || '—'),
            datasets: [{
              label: 'Casos',
              data: porOficina.map(o => o.total),
              backgroundColor: '#10b981'
            }]
          }
        });
      }
    }

    // ===== 5) Finalizado por (admins) =====
    const porAdmin = Array.isArray(stats?.porAdmin) ? stats.porAdmin : [];
    if (ctxAdmin) {
      if (porAdmin.length === 0) {
        paintMsgInChart(ctxAdmin, 'Sin datos de cierres por administrador.');
      } else {
        makeChart('porAdmin', ctxAdmin, {
          type: 'bar',
          data: {
            labels: porAdmin.map(a => a.finalizado_por || '—'),
            datasets: [{
              label: 'Casos cerrados',
              data: porAdmin.map(a => a.total),
              backgroundColor: '#8b5cf6'
            }]
          }
        });
      }
    }

  } catch (err) {
    console.error('❌ Error cargando estadísticas:', err);

    // UI de cortesía cuando falla todo el endpoint
    const kpiEl = document.getElementById('avgResolucionKpi');
    if (kpiEl) kpiEl.textContent = '—';
    paintMsgInChart(document.getElementById('chartTiempoPromedio'),
                    'No se pudieron cargar las estadísticas (500).');
    paintMsgInChart(ctxLine,    'No se pudo cargar.');
    paintMsgInChart(ctxEstado,  'No se pudo cargar.');
    paintMsgInChart(ctxTipo,    'No se pudo cargar.');
    paintMsgInChart(ctxOficina, 'No se pudo cargar.');
    paintMsgInChart(ctxAdmin,   'No se pudo cargar.');
  }
}

// logout
document.getElementById('logoutBtn')?.addEventListener('click', () => {
  clearToken();
  window.location.href = '/';
});

cargarEstadisticas();
