// /js/auth.js (FRONTEND) — versión por pestaña con sessionStorage

/**
 * Guarda token y nombre en sessionStorage.
 * Se añadió un respaldo (fallback) para evitar el mensaje 'undefined'.
 */
export function setToken(token, nombre, user) {
  sessionStorage.setItem('token', token);
  
  // 🔥 Lógica Robusta: Si 'nombre' no viene, busca en 'user'. 
  // Si ninguno existe, pone 'Administrador' por defecto.
  const nombreParaGuardar = nombre || user || 'Administrador';
  
  sessionStorage.setItem('nombre', nombreParaGuardar);
  console.log(`✅ Sesión guardada para: ${nombreParaGuardar}`);

  // Notifica a otras pestañas que hubo cambio de sesión
  try { 
    localStorage.setItem('token_sync', Date.now().toString()); 
  } catch (_) {}
}

export function getToken() {
  return sessionStorage.getItem('token');
}

/**
 * Obtiene el nombre guardado. Si no existe o es 'undefined', 
 * devuelve 'Administrador' como respaldo.
 */
export function getNombre() {
  const nombre = sessionStorage.getItem('nombre');
  if (!nombre || nombre === 'undefined') return 'Administrador';
  return nombre;
}

export function clearToken() {
  sessionStorage.removeItem('token');
  sessionStorage.removeItem('nombre');
  try { 
    localStorage.setItem('token_sync', Date.now().toString()); 
  } catch (_) {}
}

// Fetch con token
export async function fetchWithAuth(url, options = {}) {
  const token = getToken();
  if (!token) {
    window.location.href = '/';
    return;
  }
  const headers = options.headers || {};
  headers['Authorization'] = `Bearer ${token}`;
  options.headers = headers;
  return fetch(url, options);
}

// Sincronización entre pestañas
export function initAuthSync() {
  window.addEventListener('storage', (e) => {
    if (e.key === 'token_sync') {
      location.reload();
    }
  });
}