import { fetchWithAuth, clearToken } from '/js/auth.js';

const tipoEvento      = document.getElementById('tipoEvento');
const extraFields     = document.getElementById('extraFields');
const soporteForm     = document.getElementById('soporteForm');
const mensajeEl       = document.getElementById('mensaje');
const logoutBtn       = document.getElementById('logoutBtn');
const contenedorCasos = document.getElementById('misCasos');
const emptyStateEl    = document.getElementById('emptyState');

/* ==========================
   1. Campos dinámicos
========================== */
const renderSelect = (label, id, options) => `
  <div>
    <label class="block text-sm font-medium text-gray-700 mb-1">${label} <span class="text-red-500">*</span></label>
    <select id="${id}" required class="mt-1 w-full rounded-lg border-gray-300 py-2.5 px-4 focus:ring-2 focus:ring-[#007A33] focus:border-transparent shadow-sm">
      <option value="">Seleccione...</option>
      ${options.map(o => `<option value="${o}">${o}</option>`).join('')}
    </select>
  </div>
`;

tipoEvento.addEventListener('change', () => {
  const value = tipoEvento.value;
  extraFields.innerHTML = '';
  extraFields.classList.add('hidden');
  if (!value) return;

  if (value === 'Problemas en software') {
    extraFields.innerHTML = renderSelect('Programa afectado', 'equipo_select', [
      'Linix','Office','Windows','Digiturno','Bitdefender','WorkManager ED'
    ]);
  } else if (value === 'Problemas en hardware') {
    extraFields.innerHTML = renderSelect('Equipo afectado', 'equipo_select', [
      'Computador','Teclado','Mouse','Impresora','Lector de huella','Cámara PC',
      'Audífono diadema','Video Beam','Sistema de videoconferencia','Televisor',
      'Circuito cerrado de seguridad','Kiosco Autoconsulta','Escáner',
      'Puntos de conexión','Cables de red','Cables HDMI'
    ]);
  } else if (value === 'Fallas en telecomunicaciones') {
    extraFields.innerHTML = renderSelect('Servicio afectado', 'equipo_select', [
      'Celular corporativo','Teléfono (extensión)','Correo electrónico',
      'Zoom','3CX','Meet','WhatsApp','Planta Telefónica','Carpeta compartida'
    ]);
  } else if (value === 'Interrupción en el suministro de servicio público') {
    extraFields.innerHTML = renderSelect('Servicio afectado', 'equipo_select', [
      'Internet','Canal de datos'
    ]);
  }

  extraFields.classList.remove('hidden');
});

/* ==========================
   2. Enviar reporte
========================== */
soporteForm.addEventListener('submit', async (e) => {
  e.preventDefault();
  mensajeEl.classList.add('hidden');

  const equipoAfectado = (document.getElementById('equipo_select')?.value || '').trim();
  const detalleTexto   = (document.getElementById('detalle')?.value || '').trim();

  if (!tipoEvento.value)   return showMsg('❌ Debes seleccionar el tipo de evento', false);
  if (!equipoAfectado)     return showMsg('❌ Debes seleccionar el equipo/servicio afectado', false);
  if (!detalleTexto)       return showMsg('❌ El detalle del problema es obligatorio', false);

  const payload = {
    fecha: document.getElementById('fecha').value,
    oficina: document.getElementById('oficina').value || null,
    tipoEvento: tipoEvento.value,
    equipoAfectado,
    detalle: detalleTexto
  };

  try {
    const res = await fetchWithAuth('/api/reportes', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify(payload)
    });
    const data = await res.json();
    if (!res.ok) throw new Error(data.message || 'Error al guardar el reporte');

    showMsg('✅ Caso reportado con éxito', true);
    soporteForm.reset();
    extraFields.innerHTML = '';
    extraFields.classList.add('hidden');

    await cargarCasos(); // refrescar lista
  } catch (err) {
    showMsg('❌ ' + err.message, false);
  }
});

/* ==========================
   3. Helpers UI
========================== */
function showMsg(text, ok = true) {
  mensajeEl.textContent = text;
  mensajeEl.classList.remove('hidden');
  mensajeEl.classList.toggle('text-green-600', ok);
  mensajeEl.classList.toggle('text-red-600', !ok);
}

function formatearFechaHora(iso) {
  if (!iso) return '-';
  const d = new Date(iso);
  return d.toLocaleString('es-CO', {
    timeZone: 'America/Bogota',
    year: 'numeric', month: 'long', day: 'numeric',
    hour: '2-digit', minute: '2-digit', hour12: true
  });
}

function humanizarSegundos(total) {
  if (total == null) return '—';
  const s = Number(total);
  const d = Math.floor(s / 86400);
  const h = Math.floor((s % 86400) / 3600);
  const m = Math.floor((s % 3600) / 60);
  const sec = s % 60;
  const parts = [];
  if (d) parts.push(`${d}d`);
  if (h) parts.push(`${h}h`);
  if (m) parts.push(`${m}m`);
  if (!d && !h && !m) parts.push(`${sec}s`);
  return parts.join(' ');
}

/* ==========================
   4. Cargar casos del usuario
========================== */
async function cargarCasos() {
  try {
    const res = await fetchWithAuth('/api/reportes/mis-casos', {
      headers: { 'Cache-Control': 'no-cache' }
    });

    // Manejo robusto de respuesta
    let data, raw;
    try {
      raw = await res.text();
      data = raw ? JSON.parse(raw) : [];
    } catch {
      data = [];
    }
    if (!res.ok) {
      const msg = (data && data.message) ? data.message : `HTTP ${res.status}`;
      throw new Error(msg);
    }

    const casos = Array.isArray(data) ? data : [];
    console.log('[mis-casos] recibidos:', casos.length);

    contenedorCasos.innerHTML = '';

    if (casos.length === 0) {
      contenedorCasos.innerHTML = '<p class="text-gray-500">No tienes casos registrados.</p>';
      if (emptyStateEl) emptyStateEl.classList.remove('hidden');
      return;
    } else {
      if (emptyStateEl) emptyStateEl.classList.add('hidden');
    }

    casos.forEach(caso => {
      const tiempo = (caso.estado === 'Finalizado')
        ? humanizarSegundos(caso.duracion_segundos)
        : '—';

      const div = document.createElement('div');
      div.className = "p-4 bg-white rounded-xl shadow border";
      div.innerHTML = `
        <p><b>ID:</b> ${caso.id}</p>
        <p><b>Fecha de registro:</b> ${formatearFechaHora(caso.fecha_creacion)}</p>
        <p><b>Oficina:</b> ${caso.oficina || '-'}</p>
        <p><b>Tipo:</b> ${caso.tipo_evento}</p>
        <p><b>Equipo/servicio afectado:</b> ${caso.equipo_afectado || '-'}</p>
        <p><b>Detalle:</b> ${caso.detalle || '-'}</p>
        <p><b>Estado:</b>
          <span class="font-semibold ${caso.estado === 'Finalizado' ? 'text-green-600' : 'text-orange-600'}">
            ${caso.estado}
          </span>
        </p>
        <p><b>⏱ Tiempo de resolución:</b> ${tiempo}</p>
        ${caso.estado === 'Finalizado' && caso.finalizado_por ? `
          <p><b>✔ Cerrado por:</b> ${caso.finalizado_por}</p>
        ` : ''}
      `;
      contenedorCasos.appendChild(div);
    });
  } catch (err) {
    console.error('[mis-casos] error:', err);
    if (!contenedorCasos.innerHTML.trim()) {
      contenedorCasos.innerHTML = '<p class="text-red-600">No se pudo cargar tus casos. Intenta de nuevo.</p>';
    }
  }
}

/* ==========================
   5. Autoejecutar al entrar
========================== */
document.addEventListener('DOMContentLoaded', () => {
  cargarCasos();
});

/* ==========================
   6. Logout
========================== */
logoutBtn.addEventListener('click', () => {
  clearToken();
  window.location.href = '/';
});
