<?php
namespace App\Http\Controllers;

use App\Models\Ticket;
use App\Models\TicketCategory;
use App\Models\TicketPriority;
use App\Models\TicketType;
use App\Models\NexusNotification;
use App\Services\TicketService;
use Illuminate\Http\Request;

class PortalController extends Controller
{
    public function __construct(private TicketService $ticketService) {}

    public function index()
    {
        $user    = auth()->user();
        $tickets = Ticket::with(['status','priority'])
            ->where('requester_id', $user->id)
            ->latest()
            ->paginate(10);

        $stats = [
            'open'     => Ticket::where('requester_id', $user->id)->open()->count(),
            'closed'   => Ticket::where('requester_id', $user->id)->closed()->count(),
            'total'    => Ticket::where('requester_id', $user->id)->count(),
        ];

        return view('portal.index', compact('tickets', 'stats'));
    }

    public function create()
    {
        $categories = TicketCategory::active()->roots()->with('children')->get();
        $priorities = TicketPriority::active()->get();
        $types      = TicketType::active()->get();
        return view('portal.create', compact('categories', 'priorities', 'types'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title'       => 'required|string|max:255',
            'description' => 'required|string',
            'priority_id' => 'required|exists:ticket_priorities,id',
            'category_id' => 'nullable|exists:ticket_categories,id',
            'type_id'     => 'nullable|exists:ticket_types,id',
        ]);

        $ticket = $this->ticketService->create($data, auth()->id());
        return redirect()->route('portal.show', $ticket)
            ->with('success', "Tu solicitud {$ticket->number} fue registrada correctamente.");
    }

    public function show(Ticket $ticket)
    {
        abort_unless($ticket->requester_id === auth()->id(), 403);
        $ticket->load(['status','priority','category','type','comments.user','history']);
        return view('portal.show', compact('ticket'));
    }

    public function addComment(Request $request, Ticket $ticket)
    {
        abort_unless($ticket->requester_id === auth()->id(), 403);
        $request->validate(['body' => 'required|string']);
        $ticket->comments()->create([
            'user_id'     => auth()->id(),
            'body'        => $request->body,
            'is_internal' => false,
        ]);
        $ticket->addHistory('commented', 'Comentario del solicitante');
        return back()->with('success', 'Comentario agregado.');
    }

    // Encuesta de satisfacción
    public function satisfaction(Request $request, Ticket $ticket)
    {
        abort_unless($ticket->requester_id === auth()->id(), 403);
        abort_unless($ticket->status->is_final, 422);

        $request->validate([
            'score'   => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:500',
        ]);

        $ticket->update([
            'satisfaction_score'   => $request->score,
            'satisfaction_comment' => $request->comment,
        ]);

        $ticket->addHistory('satisfaction', "Satisfacción calificada: {$request->score}/5");
        return back()->with('success', '¡Gracias por tu calificación!');
    }

    // Notificaciones del usuario
    public function notifications()
    {
        $notifications = NexusNotification::forUser(auth()->id())
            ->latest()->paginate(20);
        NexusNotification::forUser(auth()->id())->unread()->update(['read_at' => now()]);
        return view('portal.notifications', compact('notifications'));
    }
}
