<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;

class Configuration extends Model
{
    protected $fillable = ['key', 'value', 'group', 'type', 'label', 'description', 'is_public'];

    protected function casts(): array {
        return ['is_public' => 'boolean'];
    }

    // Obtener un valor de configuración con cache de 60 min
    public static function get(string $key, mixed $default = null): mixed {
        return Cache::remember("config_{$key}", 3600, function () use ($key, $default) {
            $config = static::where('key', $key)->first();
            return $config ? $config->value : $default;
        });
    }

    // Guardar un valor e invalidar cache
    public static function set(string $key, mixed $value): void {
        static::updateOrCreate(
            ['key' => $key],
            ['value' => $value]
        );
        Cache::forget("config_{$key}");
    }

    // Obtener todas las configs de un grupo
    public static function group(string $group): \Illuminate\Support\Collection {
        return static::where('group', $group)->get()->keyBy('key');
    }

    // Limpiar cache de todas las configs
    public static function clearCache(): void {
        static::all()->each(fn($c) => Cache::forget("config_{$c->key}"));
    }

    public function scopeForGroup($query, string $group) {
        return $query->where('group', $group);
    }
    public function scopePublic($query) {
        return $query->where('is_public', true);
    }
}
