<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class KbArticle extends Model
{
    use SoftDeletes, LogsActivity;

    protected $fillable = [
        'title','slug','content','excerpt','category_id','author_id',
        'status','featured','tags','published_at'
    ];

    protected function casts(): array {
        return ['featured' => 'boolean', 'published_at' => 'datetime'];
    }

    protected static function boot(): void {
        parent::boot();
        static::creating(function($m) {
            $m->slug = $m->slug ?: static::uniqueSlug($m->title);
            if (!$m->excerpt && $m->content) {
                $m->excerpt = Str::limit(strip_tags($m->content), 200);
            }
        });
        static::updating(function($m) {
            if ($m->isDirty('content') && !$m->isDirty('excerpt')) {
                $m->excerpt = Str::limit(strip_tags($m->content), 200);
            }
        });
    }

    public static function uniqueSlug(string $title): string {
        $slug = Str::slug($title);
        $count = static::where('slug', 'like', $slug . '%')->count();
        return $count ? $slug . '-' . ($count + 1) : $slug;
    }

    public function category(): BelongsTo { return $this->belongsTo(KbCategory::class); }
    public function author(): BelongsTo   { return $this->belongsTo(User::class, 'author_id'); }
    public function votes(): HasMany      { return $this->hasMany(KbVote::class, 'article_id'); }

    public function scopePublished($q) { return $q->where('status', 'published'); }
    public function scopeFeatured($q)  { return $q->where('featured', true); }

    public function getTagsArrayAttribute(): array {
        return $this->tags ? array_map('trim', explode(',', $this->tags)) : [];
    }

    public function getHelpfulRateAttribute(): ?int {
        $total = $this->helpful_yes + $this->helpful_no;
        return $total > 0 ? round(($this->helpful_yes / $total) * 100) : null;
    }

    public function incrementViews(): void {
        $this->increment('views');
    }

    public function getUserVoteAttribute(): ?bool {
        if (!auth()->check()) return null;
        $vote = $this->votes()->where('user_id', auth()->id())->first();
        return $vote?->helpful;
    }

    // Búsqueda full-text simple (compatible con MySQL sin fulltext index en todos los engines)
    public static function search(string $query): \Illuminate\Database\Eloquent\Builder
    {
        $q = '%' . $query . '%';
        return static::published()
            ->where(function($builder) use ($q, $query) {
                $builder->where('title', 'like', $q)
                        ->orWhere('excerpt', 'like', $q)
                        ->orWhere('tags', 'like', $q);
            })
            ->orderByRaw("CASE WHEN title LIKE ? THEN 0 ELSE 1 END", ['%'.$query.'%'])
            ->orderByDesc('views');
    }

    public function getActivitylogOptions(): LogOptions {
        return LogOptions::defaults()->logOnly(['title','status'])->logOnlyDirty()
            ->setDescriptionForEvent(fn($e) => "Artículo '{$this->title}': {$e}");
    }
}
