<?php
namespace App\Services;

use App\Models\NexusNotification;
use App\Models\Ticket;
use App\Models\User;

class NotificationService
{
    // Ticket creado: notificar a admins/jefes y al agente asignado
    public function onTicketCreated(Ticket $ticket): void
    {
        $url   = route('tickets.show', $ticket);
        $title = "Nuevo ticket {$ticket->number}";
        $body  = $ticket->title;

        // Notificar a todos los admins y jefes
        User::role(['admin', 'superadmin', 'jefe_soporte'])
            ->where('id', '!=', $ticket->requester_id)
            ->get()
            ->each(fn($u) => NexusNotification::send($u->id, 'ticket_created', $title, $body, $url));

        // Notificar al agente asignado si existe
        if ($ticket->assignee_id && $ticket->assignee_id !== $ticket->requester_id) {
            NexusNotification::send(
                $ticket->assignee_id, 'ticket_assigned',
                "Se te asignó el ticket {$ticket->number}", $body, $url
            );
        }
    }

    // Estado cambiado: notificar al solicitante
    public function onStatusChanged(Ticket $ticket, string $oldStatus, string $newStatus): void
    {
        $url  = route('tickets.show', $ticket);
        NexusNotification::send(
            $ticket->requester_id,
            'ticket_status_changed',
            "Ticket {$ticket->number} actualizado",
            "Estado cambiado de '{$oldStatus}' a '{$newStatus}'",
            $url
        );

        if ($ticket->status->is_final) {
            NexusNotification::send(
                $ticket->requester_id,
                'ticket_closed',
                "Ticket {$ticket->number} resuelto",
                "Tu solicitud ha sido resuelta. ¿Puedes calificar la atención?",
                $url
            );
        }
    }

    // Nuevo comentario: notificar a las partes
    public function onComment(Ticket $ticket, int $commenterId, bool $isInternal): void
    {
        if ($isInternal) return; // notas internas no notifican al usuario

        $url      = route('tickets.show', $ticket);
        $commenter = User::find($commenterId);

        $notifyIds = collect([$ticket->requester_id, $ticket->assignee_id])
            ->filter()
            ->unique()
            ->reject(fn($id) => $id === $commenterId);

        foreach ($notifyIds as $uid) {
            NexusNotification::send(
                $uid, 'ticket_comment',
                "Nuevo comentario en {$ticket->number}",
                "{$commenter->name} comentó en: {$ticket->title}",
                $url
            );
        }
    }

    // Asignación: notificar al nuevo agente
    public function onAssigned(Ticket $ticket, ?int $oldAssigneeId): void
    {
        if (!$ticket->assignee_id) return;
        $url = route('tickets.show', $ticket);
        NexusNotification::send(
            $ticket->assignee_id,
            'ticket_assigned',
            "Se te asignó el ticket {$ticket->number}",
            $ticket->title,
            $url
        );
    }

    // SLA por vencer (llamado por comando/scheduler)
    public function onSlaWarning(Ticket $ticket): void
    {
        $url  = route('tickets.show', $ticket);
        $notify = collect([$ticket->assignee_id])
            ->filter()
            ->merge(User::role(['admin','jefe_soporte'])->pluck('id'));

        foreach ($notify->unique() as $uid) {
            NexusNotification::send(
                $uid, 'sla_warning',
                "SLA por vencer — {$ticket->number}",
                "Quedan menos de 2 horas para vencer el SLA de resolución",
                $url
            );
        }
    }
}
