<?php
require_once __DIR__ . '/config/config.php';
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/functions.php';

startSession();
requireLogin();
requireRole(['admin','agent']);

$page      = 'assets';
$pageTitle = 'Inventario de Activos';

// ── ACCIONES ────────────────────────────────────────────────
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    if ($action === 'create' || $action === 'edit') {
        $data = [
            'asset_tag'      => trim($_POST['asset_tag']),
            'name'           => trim($_POST['name']),
            'type'           => $_POST['type'],
            'brand'          => trim($_POST['brand'] ?? ''),
            'model'          => trim($_POST['model'] ?? ''),
            'serial_number'  => trim($_POST['serial_number'] ?? ''),
            'status'         => $_POST['status'],
            'location'       => trim($_POST['location'] ?? ''),
            'assigned_to'    => $_POST['assigned_to'] ?: null,
            'purchase_date'  => $_POST['purchase_date'] ?: null,
            'warranty_expiry'=> $_POST['warranty_expiry'] ?: null,
            'purchase_cost'  => $_POST['purchase_cost'] ?: null,
            'ip_address'     => trim($_POST['ip_address'] ?? '') ?: null,
            'mac_address'    => trim($_POST['mac_address'] ?? '') ?: null,
            'os'             => trim($_POST['os'] ?? '') ?: null,
            'notes'          => trim($_POST['notes'] ?? ''),
        ];

        if ($action === 'create') {
            $aid = db()->insert(
                "INSERT INTO assets (asset_tag,name,type,brand,model,serial_number,status,location,assigned_to,
                  purchase_date,warranty_expiry,purchase_cost,ip_address,mac_address,os,notes) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)",
                array_values($data)
            );
            db()->insert("INSERT INTO asset_history (asset_id,user_id,action,notes) VALUES (?,?,?,?)",
                [$aid, $_SESSION['user_id'], 'Activo creado', 'Registro inicial en inventario']);
            auditLog('asset_created', 'assets', $aid, null, $data);
            flash('success', 'Activo registrado exitosamente.');
        } else {
            $aid = (int)$_POST['asset_id'];
            $old = db()->fetchOne("SELECT * FROM assets WHERE id=?", [$aid]);
            db()->execute(
                "UPDATE assets SET asset_tag=?,name=?,type=?,brand=?,model=?,serial_number=?,status=?,location=?,
                 assigned_to=?,purchase_date=?,warranty_expiry=?,purchase_cost=?,ip_address=?,mac_address=?,os=?,notes=?,updated_at=NOW() WHERE id=?",
                [...array_values($data), $aid]
            );
            // Registrar cambio en historial
            $changes = [];
            foreach ($data as $k => $v) {
                if (($old[$k] ?? '') != $v) $changes[] = "$k: {$old[$k]} → $v";
            }
            if ($changes) {
                db()->insert("INSERT INTO asset_history (asset_id,user_id,action,old_value,new_value,notes) VALUES (?,?,?,?,?,?)",
                    [$aid, $_SESSION['user_id'], 'Actualización', json_encode($old), json_encode($data), implode('; ', $changes)]);
            }
            auditLog('asset_updated', 'assets', $aid, $old, $data);
            flash('success', 'Activo actualizado.');
        }
    }

    if ($action === 'maintenance') {
        $aid   = (int)$_POST['asset_id'];
        $notes = trim($_POST['notes']);
        db()->insert("INSERT INTO asset_history (asset_id,user_id,action,notes) VALUES (?,?,?,?)",
            [$aid, $_SESSION['user_id'], 'Mantenimiento registrado', $notes]);
        db()->execute("UPDATE assets SET status='maintenance', updated_at=NOW() WHERE id=?", [$aid]);
        flash('success', 'Mantenimiento registrado.');
    }

    header('Location: ' . BASE_URL . '/assets.php');
    exit;
}

// Datos
$filters = [];
$params  = [];
$search  = trim($_GET['q'] ?? '');
$type    = $_GET['type'] ?? '';
$status  = $_GET['status'] ?? '';

if ($search)  { $filters[] = '(a.name LIKE ? OR a.asset_tag LIKE ? OR a.serial_number LIKE ?)'; $params = array_merge($params, ["%$search%","%$search%","%$search%"]); }
if ($type)    { $filters[] = 'a.type = ?';   $params[] = $type; }
if ($status)  { $filters[] = 'a.status = ?'; $params[] = $status; }

$where   = $filters ? 'WHERE ' . implode(' AND ', $filters) : '';
$assets  = db()->fetchAll(
    "SELECT a.*, u.name as assigned_name FROM assets a LEFT JOIN users u ON a.assigned_to = u.id $where ORDER BY a.asset_tag", $params
);
$users   = db()->fetchAll("SELECT id, name FROM users WHERE active=1 ORDER BY name");

$typeLabels = ['computer'=>'Computador','laptop'=>'Laptop','server'=>'Servidor','printer'=>'Impresora',
               'network'=>'Red','mobile'=>'Móvil','peripheral'=>'Periférico','software'=>'Software','other'=>'Otro'];
$statusColors = ['active'=>'success','maintenance'=>'warning','retired'=>'secondary','lost'=>'danger'];

require_once __DIR__ . '/includes/header.php';
?>

<?php if (flash('success')): ?>
<div class="alert alert-success"><i class="bi bi-check-circle me-2"></i><?= flash('success') ?></div>
<?php endif; ?>

<!-- FILTROS -->
<div class="card mb-3">
    <div class="card-body py-2">
        <form method="GET" class="row g-2 align-items-center">
            <div class="col-md-3"><input type="text" name="q" class="form-control form-control-sm" placeholder="🔍 Buscar activo..." value="<?= htmlspecialchars($search) ?>"></div>
            <div class="col-auto">
                <select name="type" class="form-select form-select-sm">
                    <option value="">Todos los tipos</option>
                    <?php foreach ($typeLabels as $k=>$v): ?>
                    <option value="<?= $k ?>" <?= $type===$k?'selected':'' ?>><?= $v ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-auto">
                <select name="status" class="form-select form-select-sm">
                    <option value="">Todos los estados</option>
                    <?php foreach (['active'=>'Activo','maintenance'=>'Mantenimiento','retired'=>'Retirado','lost'=>'Perdido'] as $k=>$v): ?>
                    <option value="<?= $k ?>" <?= $status===$k?'selected':'' ?>><?= $v ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="col-auto">
                <button type="submit" class="btn btn-sm btn-primary">Filtrar</button>
                <a href="<?= BASE_URL ?>/assets.php" class="btn btn-sm btn-outline-secondary">Limpiar</a>
            </div>
            <div class="col-auto ms-auto">
                <button type="button" class="btn btn-sm btn-success" data-bs-toggle="modal" data-bs-target="#assetModal">
                    <i class="bi bi-plus"></i> Nuevo Activo
                </button>
            </div>
        </form>
    </div>
</div>

<!-- STATS -->
<div class="row g-2 mb-3">
    <?php
    $totals = db()->fetchAll("SELECT status, COUNT(*) c FROM assets GROUP BY status");
    $tMap   = array_column($totals, 'c', 'status');
    $sLabels = ['active'=>['Activos','#10B981'],'maintenance'=>['Mantenimiento','#F59E0B'],'retired'=>['Retirados','#6B7280'],'lost'=>['Perdidos','#EF4444']];
    foreach ($sLabels as $k => [$label, $color]):
    ?>
    <div class="col-6 col-md-3">
        <div class="card text-center py-2">
            <div style="font-size:1.5rem;font-weight:700;color:<?= $color ?>"><?= $tMap[$k] ?? 0 ?></div>
            <div style="font-size:.75rem;color:#6B7280;"><?= $label ?></div>
        </div>
    </div>
    <?php endforeach; ?>
</div>

<!-- TABLA DE ACTIVOS -->
<div class="card">
    <div class="card-body p-0">
        <div class="table-responsive">
            <table class="table table-hover mb-0">
                <thead class="table-light">
                    <tr>
                        <th>Tag</th>
                        <th>Nombre</th>
                        <th>Tipo</th>
                        <th>Marca/Modelo</th>
                        <th>Serial</th>
                        <th>Estado</th>
                        <th>Asignado a</th>
                        <th>Ubicación</th>
                        <th>Garantía</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($assets as $a): ?>
                    <tr>
                        <td><code class="text-primary"><?= sanitize($a['asset_tag']) ?></code></td>
                        <td class="fw-semibold"><?= sanitize($a['name']) ?></td>
                        <td><small><?= $typeLabels[$a['type']] ?? $a['type'] ?></small></td>
                        <td><small><?= sanitize($a['brand'] . ' ' . $a['model']) ?></small></td>
                        <td><small class="text-muted"><?= sanitize($a['serial_number'] ?? '-') ?></small></td>
                        <td><span class="badge bg-<?= $statusColors[$a['status']] ?> text-<?= $a['status']=='maintenance'?'dark':'' ?>"><?= ucfirst($a['status']) ?></span></td>
                        <td><small><?= sanitize($a['assigned_name'] ?? '-') ?></small></td>
                        <td><small><?= sanitize($a['location'] ?? '-') ?></small></td>
                        <td>
                            <?php if ($a['warranty_expiry']): ?>
                            <small class="<?= $a['warranty_expiry'] < date('Y-m-d') ? 'text-danger' : 'text-success' ?>">
                                <?= date('d/m/Y', strtotime($a['warranty_expiry'])) ?>
                            </small>
                            <?php else: ?><small class="text-muted">-</small><?php endif; ?>
                        </td>
                        <td>
                            <button class="btn btn-xs btn-sm btn-outline-primary" onclick="editAsset(<?= json_encode($a) ?>)" title="Editar">
                                <i class="bi bi-pencil"></i>
                            </button>
                            <button class="btn btn-xs btn-sm btn-outline-warning" onclick="openHistory(<?= $a['id'] ?>, '<?= sanitize($a['name']) ?>')" title="Historial">
                                <i class="bi bi-clock-history"></i>
                            </button>
                            <form method="POST" style="display:inline;" onsubmit="return confirm('¿Registrar mantenimiento?')">
                                <input type="hidden" name="action" value="maintenance">
                                <input type="hidden" name="asset_id" value="<?= $a['id'] ?>">
                                <input type="hidden" name="notes" value="Mantenimiento preventivo">
                                <button type="submit" class="btn btn-xs btn-sm btn-outline-secondary" title="Mantenimiento">
                                    <i class="bi bi-tools"></i>
                                </button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                    <?php if (empty($assets)): ?>
                    <tr><td colspan="10" class="text-center text-muted py-4">No hay activos registrados.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- MODAL NUEVO/EDITAR ACTIVO -->
<div class="modal fade" id="assetModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <form method="POST">
                <input type="hidden" name="action" id="assetAction" value="create">
                <input type="hidden" name="asset_id" id="assetId" value="">
                <div class="modal-header">
                    <h5 class="modal-title" id="assetModalTitle">Nuevo Activo</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row g-3">
                        <div class="col-md-4"><label class="form-label small fw-semibold">Tag / Código *</label>
                            <input type="text" name="asset_tag" id="f_asset_tag" class="form-control" required></div>
                        <div class="col-md-8"><label class="form-label small fw-semibold">Nombre del Activo *</label>
                            <input type="text" name="name" id="f_name" class="form-control" required></div>
                        <div class="col-md-4"><label class="form-label small fw-semibold">Tipo</label>
                            <select name="type" id="f_type" class="form-select">
                                <?php foreach ($typeLabels as $k=>$v): ?>
                                <option value="<?= $k ?>"><?= $v ?></option>
                                <?php endforeach; ?>
                            </select></div>
                        <div class="col-md-4"><label class="form-label small fw-semibold">Marca</label>
                            <input type="text" name="brand" id="f_brand" class="form-control"></div>
                        <div class="col-md-4"><label class="form-label small fw-semibold">Modelo</label>
                            <input type="text" name="model" id="f_model" class="form-control"></div>
                        <div class="col-md-4"><label class="form-label small fw-semibold">Número de Serie</label>
                            <input type="text" name="serial_number" id="f_serial" class="form-control"></div>
                        <div class="col-md-4"><label class="form-label small fw-semibold">Estado</label>
                            <select name="status" id="f_status" class="form-select">
                                <option value="active">Activo</option>
                                <option value="maintenance">Mantenimiento</option>
                                <option value="retired">Retirado</option>
                                <option value="lost">Perdido</option>
                            </select></div>
                        <div class="col-md-4"><label class="form-label small fw-semibold">Asignado a</label>
                            <select name="assigned_to" id="f_assigned" class="form-select">
                                <option value="">Sin asignar</option>
                                <?php foreach ($users as $u): ?>
                                <option value="<?= $u['id'] ?>"><?= sanitize($u['name']) ?></option>
                                <?php endforeach; ?>
                            </select></div>
                        <div class="col-md-6"><label class="form-label small fw-semibold">Ubicación</label>
                            <input type="text" name="location" id="f_location" class="form-control"></div>
                        <div class="col-md-3"><label class="form-label small fw-semibold">F. Compra</label>
                            <input type="date" name="purchase_date" id="f_purchase" class="form-control"></div>
                        <div class="col-md-3"><label class="form-label small fw-semibold">F. Garantía</label>
                            <input type="date" name="warranty_expiry" id="f_warranty" class="form-control"></div>
                        <div class="col-md-3"><label class="form-label small fw-semibold">Costo ($)</label>
                            <input type="number" step="0.01" name="purchase_cost" id="f_cost" class="form-control"></div>
                        <div class="col-md-3"><label class="form-label small fw-semibold">IP</label>
                            <input type="text" name="ip_address" id="f_ip" class="form-control" placeholder="192.168.1.x"></div>
                        <div class="col-md-3"><label class="form-label small fw-semibold">MAC</label>
                            <input type="text" name="mac_address" id="f_mac" class="form-control"></div>
                        <div class="col-md-3"><label class="form-label small fw-semibold">Sistema Operativo</label>
                            <input type="text" name="os" id="f_os" class="form-control"></div>
                        <div class="col-12"><label class="form-label small fw-semibold">Notas</label>
                            <textarea name="notes" id="f_notes" class="form-control" rows="2"></textarea></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                    <button type="submit" class="btn btn-primary">Guardar</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- MODAL HISTORIAL -->
<div class="modal fade" id="historyModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Historial - <span id="historyAssetName"></span></h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="historyContent">Cargando...</div>
        </div>
    </div>
</div>

<?php
$extraJs = '<script>
function editAsset(a) {
    document.getElementById("assetAction").value = "edit";
    document.getElementById("assetId").value = a.id;
    document.getElementById("assetModalTitle").textContent = "Editar Activo";
    ["asset_tag","name","type","brand","model","serial_number","status","location","assigned_to","ip_address","mac_address","os","notes"].forEach(f => {
        const el = document.getElementById("f_" + f.replace("_","").replace("_","").replace("_",""));
    });
    document.getElementById("f_asset_tag").value = a.asset_tag || "";
    document.getElementById("f_name").value = a.name || "";
    document.getElementById("f_type").value = a.type || "computer";
    document.getElementById("f_brand").value = a.brand || "";
    document.getElementById("f_model").value = a.model || "";
    document.getElementById("f_serial").value = a.serial_number || "";
    document.getElementById("f_status").value = a.status || "active";
    document.getElementById("f_assigned").value = a.assigned_to || "";
    document.getElementById("f_location").value = a.location || "";
    document.getElementById("f_purchase").value = a.purchase_date || "";
    document.getElementById("f_warranty").value = a.warranty_expiry || "";
    document.getElementById("f_cost").value = a.purchase_cost || "";
    document.getElementById("f_ip").value = a.ip_address || "";
    document.getElementById("f_mac").value = a.mac_address || "";
    document.getElementById("f_os").value = a.os || "";
    document.getElementById("f_notes").value = a.notes || "";
    new bootstrap.Modal(document.getElementById("assetModal")).show();
}

function openHistory(id, name) {
    document.getElementById("historyAssetName").textContent = name;
    document.getElementById("historyContent").innerHTML = "Cargando...";
    fetch("<?= BASE_URL ?>/api/asset-history.php?id=" + id)
        .then(r => r.text()).then(h => { document.getElementById("historyContent").innerHTML = h; });
    new bootstrap.Modal(document.getElementById("historyModal")).show();
}
</script>';
require_once __DIR__ . '/includes/footer.php';
?>
