<?php
require_once __DIR__ . '/db.php';

// ============================================================
// AUTENTICACIÓN
// ============================================================
function startSession() {
    if (session_status() === PHP_SESSION_NONE) {
        session_name(SESSION_NAME);
        session_start();
    }
}

function login($email, $password) {
    $user = db()->fetchOne("SELECT * FROM users WHERE email = ? AND active = 1", [$email]);
    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['user_id']   = $user['id'];
        $_SESSION['user_name'] = $user['name'];
        $_SESSION['user_role'] = $user['role'];
        $_SESSION['user_email']= $user['email'];
        db()->execute("UPDATE users SET last_login = NOW() WHERE id = ?", [$user['id']]);
        auditLog('login', 'users', $user['id']);
        return true;
    }
    return false;
}

function logout() {
    startSession();
    session_destroy();
    header('Location: ' . BASE_URL . '/index.php');
    exit;
}

function isLoggedIn() {
    startSession();
    return isset($_SESSION['user_id']);
}

function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . '/index.php?redirect=' . urlencode($_SERVER['REQUEST_URI']));
        exit;
    }
}

function requireRole($roles) {
    requireLogin();
    if (!in_array($_SESSION['user_role'], (array)$roles)) {
        header('Location: ' . BASE_URL . '/dashboard.php?error=forbidden');
        exit;
    }
}

function currentUser() {
    if (!isLoggedIn()) return null;
    return db()->fetchOne("SELECT * FROM users WHERE id = ?", [$_SESSION['user_id']]);
}

function isAdmin()  { return isLoggedIn() && $_SESSION['user_role'] === 'admin'; }
function isAgent()  { return isLoggedIn() && in_array($_SESSION['user_role'], ['admin','agent']); }

// ============================================================
// TICKETS
// ============================================================
function generateTicketNumber() {
    $year  = date('Y');
    $month = date('m');
    $last  = db()->fetchOne(
        "SELECT ticket_number FROM tickets ORDER BY id DESC LIMIT 1"
    );
    if ($last) {
        preg_match('/(\d+)$/', $last['ticket_number'], $m);
        $num = intval($m[1] ?? 0) + 1;
    } else {
        $num = 1;
    }
    return TICKET_PREFIX . '-' . $year . $month . '-' . str_pad($num, 4, '0', STR_PAD_LEFT);
}

function createTicket($data, $userId) {
    $ticketNum = generateTicketNumber();
    
    // Calcular fecha límite SLA
    $slaHours = 24;
    if (!empty($data['category_id'])) {
        $cat = db()->fetchOne("SELECT sla_hours FROM categories WHERE id = ?", [$data['category_id']]);
        if ($cat) $slaHours = $cat['sla_hours'];
    }
    $dueDate = date('Y-m-d H:i:s', strtotime("+{$slaHours} hours"));

    $id = db()->insert(
        "INSERT INTO tickets (ticket_number, title, description, priority, category_id, subcategory_id, requester_id, asset_id, due_date) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)",
        [
            $ticketNum,
            sanitize($data['title']),
            sanitize($data['description']),
            $data['priority'] ?? 'medium',
            $data['category_id'] ?? null,
            $data['subcategory_id'] ?? null,
            $userId,
            $data['asset_id'] ?? null,
            $dueDate
        ]
    );

    auditLog('ticket_created', 'tickets', $id, null, ['ticket_number' => $ticketNum]);
    sendTicketNotification($id, 'created');
    notifyUser($userId, 'Ticket creado exitosamente', "Tu ticket {$ticketNum} ha sido creado.", 'ticket', $id);
    
    return $id;
}

function updateTicketStatus($ticketId, $status, $userId) {
    $ticket = db()->fetchOne("SELECT * FROM tickets WHERE id = ?", [$ticketId]);
    $extra  = [];
    if ($status === 'resolved') $extra = [date('Y-m-d H:i:s'), null];
    if ($status === 'closed')   $extra = [$ticket['resolved_at'] ?? date('Y-m-d H:i:s'), date('Y-m-d H:i:s')];

    if ($status === 'resolved') {
        db()->execute("UPDATE tickets SET status=?, resolved_at=?, updated_at=NOW() WHERE id=?", [$status, $extra[0], $ticketId]);
    } elseif ($status === 'closed') {
        db()->execute("UPDATE tickets SET status=?, resolved_at=?, closed_at=?, updated_at=NOW() WHERE id=?", [$status, $extra[0], $extra[1], $ticketId]);
    } else {
        db()->execute("UPDATE tickets SET status=?, updated_at=NOW() WHERE id=?", [$status, $ticketId]);
    }

    auditLog('status_changed', 'tickets', $ticketId, ['status' => $ticket['status']], ['status' => $status]);
    sendTicketNotification($ticketId, 'status_changed', ['old_status' => $ticket['status'], 'new_status' => $status]);
}

function addComment($ticketId, $userId, $message, $isInternal = false) {
    $id = db()->insert(
        "INSERT INTO ticket_comments (ticket_id, user_id, message, is_internal) VALUES (?, ?, ?, ?)",
        [$ticketId, $userId, sanitize($message), $isInternal ? 1 : 0]
    );
    if (!$isInternal) {
        sendTicketNotification($ticketId, 'comment_added');
    }
    return $id;
}

function getStatusLabel($status) {
    $map = [
        'open'        => ['Abierto',       'badge-open'],
        'in_progress' => ['En Progreso',   'badge-progress'],
        'pending'     => ['Pendiente',      'badge-pending'],
        'resolved'    => ['Resuelto',      'badge-resolved'],
        'closed'      => ['Cerrado',       'badge-closed'],
    ];
    return $map[$status] ?? [$status, 'badge-secondary'];
}

function getPriorityLabel($p) {
    $map = [
        'low'      => ['Baja',      'priority-low'],
        'medium'   => ['Media',     'priority-medium'],
        'high'     => ['Alta',      'priority-high'],
        'critical' => ['Crítica',   'priority-critical'],
    ];
    return $map[$p] ?? [$p, ''];
}

// ============================================================
// AUDITORÍA
// ============================================================
function auditLog($action, $entity, $entityId = null, $oldData = null, $newData = null) {
    $userId = $_SESSION['user_id'] ?? null;
    $ip     = $_SERVER['REMOTE_ADDR'] ?? null;
    $ua     = $_SERVER['HTTP_USER_AGENT'] ?? null;
    db()->insert(
        "INSERT INTO audit_log (user_id, action, entity, entity_id, old_data, new_data, ip_address, user_agent) VALUES (?,?,?,?,?,?,?,?)",
        [$userId, $action, $entity, $entityId,
         $oldData ? json_encode($oldData) : null,
         $newData ? json_encode($newData) : null,
         $ip, $ua]
    );
}

// ============================================================
// NOTIFICACIONES
// ============================================================
function notifyUser($userId, $title, $message, $type = 'system', $entityId = null) {
    db()->insert(
        "INSERT INTO notifications (user_id, title, message, type, entity_id) VALUES (?,?,?,?,?)",
        [$userId, $title, $message, $type, $entityId]
    );
}

function getUnreadCount($userId) {
    $row = db()->fetchOne("SELECT COUNT(*) as cnt FROM notifications WHERE user_id=? AND read_at IS NULL", [$userId]);
    return $row['cnt'] ?? 0;
}

function sendTicketNotification($ticketId, $event, $extra = []) {
    if (!MAIL_ENABLED) return;
    $ticket   = db()->fetchOne(
        "SELECT t.*, u.email as req_email, u.name as req_name, a.email as agent_email, a.name as agent_name 
         FROM tickets t 
         JOIN users u ON t.requester_id = u.id 
         LEFT JOIN users a ON t.agent_id = a.id 
         WHERE t.id = ?", [$ticketId]
    );
    if (!$ticket) return;
    
    require_once __DIR__ . '/mailer.php';
    $mailer = new Mailer();

    switch ($event) {
        case 'created':
            $mailer->send($ticket['req_email'], $ticket['req_name'],
                "Ticket {$ticket['ticket_number']} creado",
                emailTemplate('Tu ticket ha sido creado', "Ticket <strong>{$ticket['ticket_number']}</strong>: {$ticket['title']}<br>Estado: Abierto<br>Nos pondremos en contacto pronto.", $ticketId)
            );
            break;
        case 'status_changed':
            $newStatus = getStatusLabel($extra['new_status'])[0];
            $mailer->send($ticket['req_email'], $ticket['req_name'],
                "Ticket {$ticket['ticket_number']} actualizado",
                emailTemplate('Estado de ticket actualizado', "Tu ticket <strong>{$ticket['ticket_number']}</strong> cambió a estado: <strong>{$newStatus}</strong>", $ticketId)
            );
            break;
        case 'comment_added':
            $mailer->send($ticket['req_email'], $ticket['req_name'],
                "Nueva respuesta en ticket {$ticket['ticket_number']}",
                emailTemplate('Nueva respuesta', "Hay una nueva respuesta en tu ticket <strong>{$ticket['ticket_number']}</strong>.", $ticketId)
            );
            break;
    }
}

function emailTemplate($subject, $body, $ticketId = null) {
    $link = $ticketId ? '<p><a href="' . BASE_URL . '/ticket-view.php?id=' . $ticketId . '" style="background:#3B82F6;color:#fff;padding:10px 20px;border-radius:5px;text-decoration:none;">Ver Ticket</a></p>' : '';
    return '<!DOCTYPE html><html><body style="font-family:Arial,sans-serif;background:#f4f4f4;padding:20px;">
    <div style="max-width:600px;margin:0 auto;background:#fff;border-radius:8px;overflow:hidden;">
    <div style="background:#1E3A5F;padding:20px;text-align:center;"><h2 style="color:#fff;margin:0;">' . SITE_NAME . '</h2></div>
    <div style="padding:30px;"><h3>' . $subject . '</h3><p>' . $body . '</p>' . $link . '
    <hr><p style="color:#999;font-size:12px;">Este es un mensaje automático de ' . SITE_COMPANY . '. Por favor no responda este correo.</p>
    </div></div></body></html>';
}

// ============================================================
// UTILIDADES
// ============================================================
function sanitize($str) {
    return htmlspecialchars(trim($str), ENT_QUOTES, 'UTF-8');
}

function timeAgo($datetime) {
    $diff = time() - strtotime($datetime);
    if ($diff < 60)     return 'Hace ' . $diff . 's';
    if ($diff < 3600)   return 'Hace ' . floor($diff/60) . 'm';
    if ($diff < 86400)  return 'Hace ' . floor($diff/3600) . 'h';
    return 'Hace ' . floor($diff/86400) . 'd';
}

function formatDate($d) {
    if (!$d) return '-';
    return date('d/m/Y H:i', strtotime($d));
}

function redirect($url) {
    header('Location: ' . BASE_URL . '/' . $url);
    exit;
}

function flash($key, $msg = null) {
    if ($msg !== null) {
        $_SESSION['flash'][$key] = $msg;
    } else {
        $val = $_SESSION['flash'][$key] ?? null;
        unset($_SESSION['flash'][$key]);
        return $val;
    }
}
