-- ============================================================
-- HELPDESK ITSM - Script de instalación de base de datos
-- Compatible con MySQL 5.7+ / MariaDB 10.3+
-- ============================================================

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- --------------------------------------------------------
-- USUARIOS DEL SISTEMA
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `users` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `name` VARCHAR(120) NOT NULL,
  `email` VARCHAR(180) NOT NULL UNIQUE,
  `password` VARCHAR(255) NOT NULL,
  `role` ENUM('admin','agent','client') DEFAULT 'client',
  `department` VARCHAR(100) DEFAULT NULL,
  `phone` VARCHAR(30) DEFAULT NULL,
  `avatar` VARCHAR(255) DEFAULT NULL,
  `active` TINYINT(1) DEFAULT 1,
  `last_login` DATETIME DEFAULT NULL,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- CATEGORÍAS Y SUBCATEGORÍAS DE TICKETS
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `categories` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `name` VARCHAR(120) NOT NULL,
  `description` TEXT,
  `color` VARCHAR(7) DEFAULT '#3B82F6',
  `icon` VARCHAR(50) DEFAULT 'bi-tag',
  `parent_id` INT DEFAULT NULL,
  `sla_hours` INT DEFAULT 24,
  `active` TINYINT(1) DEFAULT 1,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (`parent_id`) REFERENCES `categories`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- TICKETS
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `tickets` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `ticket_number` VARCHAR(20) NOT NULL UNIQUE,
  `title` VARCHAR(255) NOT NULL,
  `description` TEXT NOT NULL,
  `status` ENUM('open','in_progress','pending','resolved','closed') DEFAULT 'open',
  `priority` ENUM('low','medium','high','critical') DEFAULT 'medium',
  `category_id` INT DEFAULT NULL,
  `subcategory_id` INT DEFAULT NULL,
  `requester_id` INT NOT NULL,
  `agent_id` INT DEFAULT NULL,
  `asset_id` INT DEFAULT NULL,
  `due_date` DATETIME DEFAULT NULL,
  `resolved_at` DATETIME DEFAULT NULL,
  `closed_at` DATETIME DEFAULT NULL,
  `satisfaction` TINYINT DEFAULT NULL,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (`category_id`) REFERENCES `categories`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`subcategory_id`) REFERENCES `categories`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`requester_id`) REFERENCES `users`(`id`),
  FOREIGN KEY (`agent_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- COMENTARIOS / RESPUESTAS DE TICKETS
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `ticket_comments` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `ticket_id` INT NOT NULL,
  `user_id` INT NOT NULL,
  `message` TEXT NOT NULL,
  `is_internal` TINYINT(1) DEFAULT 0,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (`ticket_id`) REFERENCES `tickets`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- ADJUNTOS DE TICKETS
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `ticket_attachments` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `ticket_id` INT NOT NULL,
  `comment_id` INT DEFAULT NULL,
  `user_id` INT NOT NULL,
  `filename` VARCHAR(255) NOT NULL,
  `original_name` VARCHAR(255) NOT NULL,
  `file_size` INT DEFAULT 0,
  `file_type` VARCHAR(100) DEFAULT NULL,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (`ticket_id`) REFERENCES `tickets`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- INVENTARIO DE ACTIVOS
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `assets` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `asset_tag` VARCHAR(50) NOT NULL UNIQUE,
  `name` VARCHAR(150) NOT NULL,
  `type` ENUM('computer','laptop','server','printer','network','mobile','peripheral','software','other') DEFAULT 'computer',
  `brand` VARCHAR(100) DEFAULT NULL,
  `model` VARCHAR(100) DEFAULT NULL,
  `serial_number` VARCHAR(100) DEFAULT NULL,
  `status` ENUM('active','maintenance','retired','lost') DEFAULT 'active',
  `location` VARCHAR(150) DEFAULT NULL,
  `assigned_to` INT DEFAULT NULL,
  `purchase_date` DATE DEFAULT NULL,
  `warranty_expiry` DATE DEFAULT NULL,
  `purchase_cost` DECIMAL(10,2) DEFAULT NULL,
  `ip_address` VARCHAR(45) DEFAULT NULL,
  `mac_address` VARCHAR(17) DEFAULT NULL,
  `os` VARCHAR(100) DEFAULT NULL,
  `notes` TEXT,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (`assigned_to`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- HISTORIAL DE ACTIVOS (Mantenimientos / Cambios)
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `asset_history` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `asset_id` INT NOT NULL,
  `user_id` INT NOT NULL,
  `action` VARCHAR(100) NOT NULL,
  `old_value` TEXT,
  `new_value` TEXT,
  `notes` TEXT,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (`asset_id`) REFERENCES `assets`(`id`) ON DELETE CASCADE,
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- BASE DE CONOCIMIENTO
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `knowledge_base` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `title` VARCHAR(255) NOT NULL,
  `content` LONGTEXT NOT NULL,
  `category_id` INT DEFAULT NULL,
  `author_id` INT NOT NULL,
  `views` INT DEFAULT 0,
  `helpful_yes` INT DEFAULT 0,
  `helpful_no` INT DEFAULT 0,
  `published` TINYINT(1) DEFAULT 1,
  `tags` VARCHAR(500) DEFAULT NULL,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
  FOREIGN KEY (`category_id`) REFERENCES `categories`(`id`) ON DELETE SET NULL,
  FOREIGN KEY (`author_id`) REFERENCES `users`(`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- LOG DE AUDITORÍA / TRAZABILIDAD
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `audit_log` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `user_id` INT DEFAULT NULL,
  `action` VARCHAR(150) NOT NULL,
  `entity` VARCHAR(50) NOT NULL,
  `entity_id` INT DEFAULT NULL,
  `old_data` JSON DEFAULT NULL,
  `new_data` JSON DEFAULT NULL,
  `ip_address` VARCHAR(45) DEFAULT NULL,
  `user_agent` VARCHAR(500) DEFAULT NULL,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- --------------------------------------------------------
-- NOTIFICACIONES
-- --------------------------------------------------------
CREATE TABLE IF NOT EXISTS `notifications` (
  `id` INT AUTO_INCREMENT PRIMARY KEY,
  `user_id` INT NOT NULL,
  `title` VARCHAR(200) NOT NULL,
  `message` TEXT NOT NULL,
  `type` ENUM('ticket','asset','system','knowledge') DEFAULT 'system',
  `entity_id` INT DEFAULT NULL,
  `read_at` DATETIME DEFAULT NULL,
  `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;

-- ============================================================
-- DATOS INICIALES
-- ============================================================

-- Admin por defecto: admin@helpdesk.com / Admin@2025*
INSERT INTO `users` (`name`, `email`, `password`, `role`, `department`) VALUES
('Administrador', 'admin@helpdesk.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin', 'TI'),
('Agente Soporte', 'agente@helpdesk.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'agent', 'Soporte Técnico'),
('Usuario Demo', 'usuario@helpdesk.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'client', 'Ventas');

-- Nota: password para todos los usuarios demo es "password"

-- Categorías principales
INSERT INTO `categories` (`name`, `description`, `color`, `icon`, `sla_hours`) VALUES
('Hardware', 'Problemas con equipos físicos', '#EF4444', 'bi-pc-display', 4),
('Software', 'Problemas con aplicaciones y sistemas', '#3B82F6', 'bi-window', 8),
('Red y Conectividad', 'Problemas de red, internet y VPN', '#F59E0B', 'bi-wifi', 2),
('Accesos y Seguridad', 'Contraseñas, permisos y accesos', '#8B5CF6', 'bi-shield-lock', 4),
('Solicitudes', 'Solicitudes de nuevos servicios o equipos', '#10B981', 'bi-clipboard-check', 48);

-- Subcategorías
INSERT INTO `categories` (`name`, `description`, `color`, `parent_id`, `sla_hours`) VALUES
('Computador no enciende', 'Problemas de arranque', '#EF4444', 1, 2),
('Teclado/Mouse dañado', 'Periféricos físicos', '#EF4444', 1, 8),
('Monitor sin imagen', 'Problemas de pantalla', '#EF4444', 1, 4),
('Aplicación no abre', 'Errores de software', '#3B82F6', 2, 8),
('Actualización requerida', 'Actualizaciones de software', '#3B82F6', 2, 24),
('Sin internet', 'Pérdida de conectividad', '#F59E0B', 3, 1),
('VPN no conecta', 'Problemas de VPN', '#F59E0B', 3, 2),
('Cambio de contraseña', 'Reset de credenciales', '#8B5CF6', 4, 1),
('Nuevo acceso requerido', 'Solicitud de accesos', '#8B5CF6', 4, 24),
('Nuevo equipo', 'Solicitud de hardware nuevo', '#10B981', 5, 72);

-- Artículos de base de conocimiento
INSERT INTO `knowledge_base` (`title`, `content`, `category_id`, `author_id`, `published`, `tags`) VALUES
('¿Cómo restablecer mi contraseña?', '<h3>Pasos para restablecer tu contraseña</h3><p>Si olvidaste tu contraseña, sigue estos pasos:</p><ol><li>Ve a la página de inicio de sesión.</li><li>Haz clic en "¿Olvidaste tu contraseña?".</li><li>Ingresa tu correo electrónico corporativo.</li><li>Revisa tu bandeja de entrada y sigue el enlace.</li><li>Crea una nueva contraseña segura (mínimo 8 caracteres, mayúsculas, números y símbolos).</li></ol><p><strong>Recuerda:</strong> Las contraseñas expiran cada 90 días por política de seguridad.</p>', 4, 1, 1, 'contraseña,acceso,login,reset'),
('Solución a problemas de conexión VPN', '<h3>Guía de solución de problemas VPN</h3><p>Si no puedes conectarte a la VPN, prueba lo siguiente:</p><ol><li><strong>Verificar internet:</strong> Asegúrate de tener conexión a internet.</li><li><strong>Reiniciar el cliente VPN:</strong> Cierra y vuelve a abrir la aplicación.</li><li><strong>Verificar credenciales:</strong> Confirma que tu usuario y contraseña sean correctos.</li><li><strong>Cambiar servidor:</strong> Prueba seleccionando un servidor diferente.</li><li><strong>Reiniciar el equipo:</strong> A veces un reinicio resuelve el problema.</li></ol><p>Si el problema persiste, crea un ticket con el error exacto que ves en pantalla.</p>', 3, 1, 1, 'vpn,red,conexión,remoto'),
('Configuración inicial de correo corporativo', '<h3>Configurar tu correo en Outlook</h3><p>Sigue estos pasos para configurar tu cuenta de correo:</p><h4>En Outlook Desktop:</h4><ol><li>Abre Outlook y ve a Archivo > Agregar cuenta.</li><li>Ingresa tu correo corporativo.</li><li>Selecciona "Configuración manual" si se te solicita.</li><li>Usa los servidores proporcionados por TI.</li><li>Acepta el certificado si aparece la advertencia.</li></ol><h4>En dispositivo móvil:</h4><p>Descarga Outlook desde la tienda de apps e ingresa tus credenciales corporativas.</p>', 2, 1, 1, 'correo,outlook,email,configuración');

-- Activos demo
INSERT INTO `assets` (`asset_tag`, `name`, `type`, `brand`, `model`, `serial_number`, `status`, `location`, `assigned_to`, `purchase_date`, `warranty_expiry`) VALUES
('TI-001', 'Computador Gerencia', 'computer', 'Dell', 'OptiPlex 7090', 'DL7090-001', 'active', 'Oficina Piso 2', 2, '2023-01-15', '2026-01-15'),
('TI-002', 'Laptop Ventas 01', 'laptop', 'Lenovo', 'ThinkPad E14', 'LN-E14-002', 'active', 'Sala de Ventas', 3, '2022-06-10', '2025-06-10'),
('TI-003', 'Servidor Principal', 'server', 'HP', 'ProLiant DL380', 'HP-DL380-003', 'active', 'Data Center', 1, '2021-03-20', '2027-03-20'),
('TI-004', 'Impresora Red', 'printer', 'HP', 'LaserJet Pro M428', 'HP-MFP-004', 'active', 'Área Común Piso 1', NULL, '2022-11-05', '2025-11-05');

SET FOREIGN_KEY_CHECKS = 1;
